<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';

/* === RBAC: incluir e exigir permissão para EDITAR clientes === */
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('clientes','update');

// DEBUG: Verificar user_id na sessão (MANTENHA ESTA LINHA TEMPORARIAMENTE)
error_log("DEBUG - editar.php: User ID na sessão: " . ($_SESSION['user_id'] ?? 'NÃO DEFINIDO'));

// Pega o ID do cliente pela URL
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($id <= 0) {
    header("Location: " . BASE_URL . "/clientes");
    exit;
}

// Busca o cliente no banco (dados originais)
try {
    $cliente_original = fetch_one("SELECT * FROM clients WHERE id = ?", [$id], true);
    if (!$cliente_original) {
        header("Location: " . BASE_URL . "/clientes");
        exit;
    }

    /* === RBAC: se o escopo de UPDATE for 'own', só permite editar se for dono === */
    $scopeUpdate = rbac_scope('clientes','update'); // 'none' | 'own' | 'all'
    if ($scopeUpdate === 'own') {
        $me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
        $owner = isset($cliente_original['owner_user_id'])
            ? (int)$cliente_original['owner_user_id']
            : (int)($cliente_original['created_by'] ?? 0);
        if ($owner !== $me_id) {
            http_response_code(403);
            exit('Você não tem permissão para editar este cliente.');
        }
    }

} catch (Exception $e) {
    die("Erro ao carregar cliente: " . $e->getMessage());
}

$page_title = "Editar Cliente: " . htmlspecialchars($cliente_original['nome_completo']);

$erros = [];
$sucesso = null;

// Atualiza o cadastro
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    error_log("DEBUG - editar.php: Requisição POST detectada para edição do cliente ID: $id.");

    $nome_completo       = trim($_POST['nome_completo']);
    $cpf_cnpj            = trim($_POST['cpf_cnpj'] ?? '');
    $email               = trim($_POST['email'] ?? '');
    $telefone_principal  = trim($_POST['telefone_principal'] ?? '');
    $tipo_cliente        = $_POST['tipo_cliente'] ?? 'comprador';
    $endereco            = trim($_POST['endereco'] ?? '');
    $bairro              = trim($_POST['bairro'] ?? '');
    $cidade              = trim($_POST['cidade'] ?? '');
    $uf                  = trim($_POST['uf'] ?? '');
    $cep                 = trim($_POST['cep'] ?? '');
    $observacoes         = trim($_POST['observacoes'] ?? '');
    $data_nascimento     = $_POST['data_nascimento'] ?? null;
    $canal_captacao      = trim($_POST['canal_captacao'] ?? '');

    // normaliza canal (somente opções válidas)
    $opcoes_canal = ['Instagram','Indicação','Orgânico','PAP'];
    if (!in_array($canal_captacao, $opcoes_canal, true)) {
        $canal_captacao = null;
    }

    // Validações básicas
    if (empty($nome_completo)) {
        $erros[] = "O nome completo é obrigatório.";
    }
    if (empty($telefone_principal)) {
        $erros[] = "O telefone principal é obrigatório.";
    }
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $erros[] = "E-mail inválido.";
    }

    if (empty($erros)) {
        error_log("DEBUG - editar.php: Nenhum erro de validação. Tentando atualizar cliente ID: $id.");
        try {
            execute(
                "UPDATE clients 
                 SET nome_completo=?, cpf_cnpj=?, email=?, telefone_principal=?, 
                     tipo_cliente=?, endereco=?, bairro=?, cidade=?, uf=?, cep=?, 
                     observacoes=?, data_nascimento=?, canal_captacao=?, updated_at=NOW() 
                 WHERE id=?",
                [
                    $nome_completo,
                    $cpf_cnpj ?: null,
                    $email ?: null,
                    $telefone_principal,
                    $tipo_cliente,
                    $endereco ?: null,
                    $bairro ?: null,
                    $cidade ?: null,
                    $uf ?: null,
                    $cep ?: null,
                    $observacoes ?: null,
                    $data_nascimento ?: null,
                    $canal_captacao, // NOVO
                    $id
                ]
            );
            $sucesso = "Cliente atualizado com sucesso!";
            error_log("DEBUG - editar.php: Cliente ID $id ATUALIZADO com sucesso no banco.");
            
            // Recarrega os dados do cliente após a atualização
            $cliente_atualizado = fetch_one("SELECT * FROM clients WHERE id = ?", [$id], true);

            // --- LOG de EDIÇÃO ---
            $alteracoes = [];
            $campos_para_comparar = [
                'nome_completo', 'cpf_cnpj', 'email', 'telefone_principal', 
                'tipo_cliente', 'endereco', 'bairro', 'cidade', 
                'uf', 'cep', 'observacoes', 'data_nascimento', 'canal_captacao' // NOVO
            ];

            foreach ($campos_para_comparar as $campo) {
                $valor_original = $cliente_original[$campo] ?? '';
                $valor_novo     = $cliente_atualizado[$campo] ?? '';
                if ($valor_original === null) $valor_original = '';
                if ($valor_novo === null)     $valor_novo     = '';
                if ($valor_original != $valor_novo) {
                    $alteracoes[] = ucfirst(str_replace('_', ' ', $campo)) . ": de '{$valor_original}' para '{$valor_novo}'";
                }
            }

            if (!empty($alteracoes)) {
                $detalhes_log = "Cliente '{$cliente_original['nome_completo']}' editado. Alterações: " . implode("; ", $alteracoes);
                error_log("DEBUG - editar.php: Alterações detectadas para cliente ID $id. Tentando gravar log: $detalhes_log");
                try {
                    execute(
                        "INSERT INTO client_logs (client_id, user_id, acao, detalhes) VALUES (?, ?, 'editado', ?)",
                        [
                            $id,
                            $_SESSION['user_id'] ?? null,
                            $detalhes_log
                        ]
                    );
                    error_log("DEBUG - editar.php: Log de edição GRAVADO com sucesso para cliente ID $id.");
                } catch (Exception $e) {
                    error_log("DEBUG - editar.php: ERRO ao gravar log de edição para cliente ID $id: " . $e->getMessage());
                }
            } else {
                error_log("DEBUG - editar.php: Nenhuma alteração detectada para cliente ID $id. Log de edição não gravado.");
            }
            // --- Fim LOG ---

            $cliente = $cliente_atualizado;

        } catch (Exception $e) {
            $erros[] = "Erro ao atualizar: " . $e->getMessage();
            error_log("DEBUG - editar.php: ERRO geral ao atualizar cliente ID $id: " . $e->getMessage());
            $cliente = $_POST; 
            $cliente['id'] = $id;
        }
    } else {
        error_log("DEBUG - editar.php: Erros de validação encontrados para cliente ID $id.");
        $cliente = $_POST;
        $cliente['id'] = $id;
    }
} else {
    error_log("DEBUG - editar.php: Requisição GET para cliente ID: $id. Carregando dados originais.");
    $cliente = $cliente_original;
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<style>
.form-text { color:#adb5bd!important; font-size:0.875rem; }
.form-label { color:#e9ecef; font-weight:500; }
.form-control::placeholder { color:#6c757d; opacity:1; }
</style>

<div class="container mt-4 mb-5">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><i class="bi bi-pencil me-2"></i>Editar Cliente</h4>
                </div>
                <div class="card-body">
                    <?php if (!empty($erros)): ?>
                        <div class="alert alert-danger">
                            <h6><i class="bi bi-exclamation-triangle me-2"></i>Erros encontrados:</h6>
                            <ul class="mb-0">
                                <?php foreach ($erros as $erro): ?>
                                    <li><?=htmlspecialchars($erro)?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($sucesso): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <i class="bi bi-check-circle me-2"></i><?=$sucesso?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="nome_completo">Nome Completo *</label>
                                <input type="text" class="form-control" name="nome_completo"
                                       value="<?=htmlspecialchars($cliente['nome_completo'])?>" required>
                                <div class="form-text">Nome completo do cliente</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="cpf_cnpj">CPF/CNPJ</label>
                                <input type="text" class="form-control" name="cpf_cnpj"
                                       value="<?=htmlspecialchars($cliente['cpf_cnpj'])?>">
                                <div class="form-text">Opcional - para identificação fiscal</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="email">E-mail</label>
                                <input type="email" class="form-control" name="email"
                                       value="<?=htmlspecialchars($cliente['email'])?>">
                                <div class="form-text">Opcional - para contato</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="telefone_principal">Telefone Principal *</label>
                                <input type="tel" class="form-control" name="telefone_principal"
                                       value="<?=htmlspecialchars($cliente['telefone_principal'])?>" required>
                                <div class="form-text">Telefone ou WhatsApp principal</div>
                            </div>

                            <!-- Canal de Captação (NOVO) -->
                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="canal_captacao">Canal de Captação</label>
                                <select class="form-select" name="canal_captacao" id="canal_captacao">
                                    <?php
                                      $opcoes = [''=>'Selecione...','Instagram'=>'Instagram','Indicação'=>'Indicação','Orgânico'=>'Orgânico','PAP'=>'PAP'];
                                      $atual = $cliente['canal_captacao'] ?? '';
                                      foreach ($opcoes as $val=>$lbl) {
                                          $v = $val === '' ? '' : $val;
                                          $sel = ($atual === $v) ? 'selected' : '';
                                          echo "<option value=\"".htmlspecialchars($v)."\" $sel>".htmlspecialchars($lbl)."</option>";
                                      }
                                    ?>
                                </select>
                                <div class="form-text">De onde este cliente veio</div>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label">Tipo de Cliente *</label>
                                <select class="form-select" name="tipo_cliente" required>
                                    <?php
                                    $tipos = ['comprador','vendedor','locatario','proprietario'];
                                    foreach ($tipos as $t) {
                                        $sel = ($cliente['tipo_cliente'] == $t) ? 'selected' : '';
                                        echo "<option value='$t' $sel>" . ucfirst($t) . "</option>";
                                    }
                                    ?>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label" for="data_nascimento">Data de Nascimento</label>
                                <input type="date" class="form-control" name="data_nascimento"
                                       value="<?=htmlspecialchars($cliente['data_nascimento'])?>">
                            </div>

                            <div class="col-12 mb-3">
                                <label class="form-label" for="endereco">Endereço</label>
                                <input type="text" class="form-control" name="endereco"
                                       value="<?=htmlspecialchars($cliente['endereco'])?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label" for="bairro">Bairro</label>
                                <input type="text" class="form-control" name="bairro"
                                       value="<?=htmlspecialchars($cliente['bairro'])?>">
                            </div>

                            <div class="col-md-5 mb-3">
                                <label class="form-label" for="cidade">Cidade</label>
                                <input type="text" class="form-control" name="cidade"
                                       value="<?=htmlspecialchars($cliente['cidade'])?>">
                            </div>

                            <div class="col-md-3 mb-3">
                                <label class="form-label" for="uf">UF</label>
                                <input type="text" class="form-control" maxlength="2" name="uf"
                                       value="<?=htmlspecialchars($cliente['uf'])?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label" for="cep">CEP</label>
                                <input type="text" class="form-control" name="cep"
                                       value="<?=htmlspecialchars($cliente['cep'])?>">
                            </div>

                            <div class="col-12 mb-3">
                                <label class="form-label" for="observacoes">Observações</label>
                                <textarea class="form-control" rows="3" name="observacoes"><?=htmlspecialchars($cliente['observacoes'])?></textarea>
                            </div>
                        </div>

                        <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                            <a href="<?=BASE_URL?>/clientes" class="btn btn-secondary me-md-2">
                                <i class="bi bi-arrow-left me-2"></i>Voltar
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle me-2"></i>Salvar Alterações
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Máscaras reutilizadas (sem telefone secundário)
document.addEventListener('DOMContentLoaded', function() {
    const cpfCnpjInput = document.querySelector('[name="cpf_cnpj"]');
    const tel1 = document.querySelector('[name="telefone_principal"]');
    const cep = document.querySelector('[name="cep"]');

    if (cpfCnpjInput) {
        cpfCnpjInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length <= 11) {
                value = value.replace(/(\d{3})(\d)/, '$1.$2');
                value = value.replace(/(\d{3})(\d)/, '$1.$2');
                value = value.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
            } else {
                value = value.replace(/^(\d{2})(\d)/, '$1.$2');
                value = value.replace(/^(\d{2}\.\d{3})(\d)/, '$1.$2');
                value = value.replace(/^(\d{2}\.\d{3}\.\d{3})(\d)/, '$1/$2');
                value = value.replace(/^(\d{2}\.\d{3}\.\d{3}\/\d{4})(\d{2})$/, '$1-$2');
            }
            e.target.value = value;
        });
    }

    const maskPhone = (input) => {
        input.addEventListener('input', e => {
            let v = e.target.value.replace(/\D/g,'');
            v = v.replace(/(\d{2})(\d)/, '($1) $2');
            v = v.replace(/(\d{5})(\d)/, '$1-$2');
            e.target.value = v.substring(0, 15);
        });
    };
    if (tel1) maskPhone(tel1);

    if (cep) cep.addEventListener('input', e => {
        let v = e.target.value.replace(/\D/g,'');
        v = v.replace(/(\d{5})(\d)/, '$1-$2');
        e.target.value = v.substring(0, 9);
    });
});
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>