<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Verificar login
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: login.php");
    exit;
}

// Carregar funções do banco
require_once __DIR__ . '/app/db.php';

// Variáveis de busca
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$status_filter = isset($_GET['status']) ? trim($_GET['status']) : 'todos';

// Construir query
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(titulo LIKE ? OR cidade LIKE ?)";
    $params[] = "%{$search}%";
    $params[] = "%{$search}%";
}

if ($status_filter !== 'todos' && $status_filter !== '') {
    $where_conditions[] = "status = ?";
    $params[] = $status_filter;
}

$where_sql = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Buscar imóveis
$total_imoveis = 0;
$imoveis = [];
$erro = null;

try {
    // Contar total
    $count_sql = "SELECT COUNT(*) as total FROM properties {$where_sql}";
    $result = fetch_one($count_sql, $params, true);
    $total_imoveis = $result ? (int)$result['total'] : 0;
    
    // Buscar registros
    $list_sql = "SELECT * FROM properties {$where_sql} ORDER BY created_at DESC LIMIT 50";
    $imoveis = fetch_all($list_sql, $params, true);
    
} catch (Exception $e) {
    $erro = $e->getMessage();
}

// Função auxiliar para badge de status
function get_status_badge_class($status) {
    switch ($status) {
        case 'publicado':
            return 'bg-success';
        case 'rascunho':
            return 'bg-warning text-dark';
        case 'vendido':
            return 'bg-secondary';
        case 'alugado':
            return 'bg-info text-dark';
        case 'negociacao':
            return 'bg-primary';
        default:
            return 'bg-secondary';
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Imóveis - Troya CRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --bg-dark: #0d1117;
            --bg-card: #161b22;
            --bg-card-hover: #21262d;
            --border: #30363d;
            --text: #e6edf3;
            --text-muted: #8b949e;
            --primary: #238636;
        }
        body { 
            background: var(--bg-dark); 
            color: var(--text); 
            font-family: system-ui, -apple-system, sans-serif;
            min-height: 100vh;
        }
        .navbar { 
            background: var(--bg-card) !important; 
            border-bottom: 1px solid var(--border); 
            padding: 1rem 0;
        }
        .navbar-brand { 
            font-weight: 600; 
            color: var(--text) !important; 
        }
        .card { 
            background: var(--bg-card); 
            border: 1px solid var(--border); 
            color: var(--text); 
            border-radius: 8px;
        }
        .card-body { 
            padding: 1.25rem; 
        }
        .table-dark { 
            --bs-table-bg: transparent; 
            --bs-table-border-color: var(--border); 
            color: var(--text); 
        }
        .table-dark th { 
            background: var(--bg-card-hover); 
            border-color: var(--border); 
            color: var(--text); 
            font-weight: 600; 
            padding: 1rem;
        }
        .table-dark td { 
            border-color: var(--border); 
            vertical-align: middle; 
            padding: 1rem;
        }
        .table-hover tbody tr:hover { 
            background: var(--bg-card-hover); 
        }
        .btn { 
            border-radius: 6px; 
            font-weight: 500; 
            padding: 0.5rem 1rem; 
        }
        .btn-primary { 
            background: var(--primary); 
            border-color: var(--primary); 
        }
        .btn-primary:hover { 
            background: #2ea043; 
            border-color: #2ea043; 
        }
        .btn-success { 
            background: var(--primary); 
            border-color: var(--primary); 
        }
        .btn-success:hover { 
            background: #2ea043; 
        }
        .text-muted { 
            color: var(--text-muted) !important; 
        }
        .empty-state { 
            text-align: center; 
            padding: 4rem 2rem; 
        }
        .empty-state i { 
            color: var(--text-muted); 
            font-size: 5rem; 
            margin-bottom: 1.5rem; 
            opacity: 0.5; 
        }
        .form-control, .form-select { 
            background: #0d1117; 
            border: 1px solid var(--border); 
            color: var(--text); 
        }
        .form-control:focus, .form-select:focus {
            background: #0d1117;
            border-color: var(--primary);
            color: var(--text);
            box-shadow: 0 0 0 0.2rem rgba(35, 134, 54, 0.25);
        }
        .form-control::placeholder {
            color: var(--text-muted);
        }
    </style>
</head>
<body>
    <!-- NAVBAR -->
    <nav class="navbar navbar-dark">
        <div class="container-fluid">
            <a class="navbar-brand fw-bold" href="dashboard.php">
                <i class="bi bi-arrow-left me-2"></i>Troya CRM
            </a>
            <div class="d-flex align-items-center gap-3">
                <span class="text-white"><?php echo htmlspecialchars($_SESSION["user_name"] ?? 'Usuário'); ?></span>
                <a href="logout.php" class="btn btn-outline-light btn-sm">
                    <i class="bi bi-box-arrow-right me-1"></i>Sair
                </a>
            </div>
        </div>
    </nav>

    <div class="container mt-4 mb-5">
        <!-- HEADER -->
        <div class="d-flex justify-content-between align-items-center mb-4 flex-wrap gap-3">
            <div>
                <h1 class="mb-1 fw-bold">
                    <i class="bi bi-house-door me-2 text-success"></i>
                    Imóveis
                    <?php if ($total_imoveis > 0): ?>
                        <span class="badge bg-secondary"><?php echo $total_imoveis; ?></span>
                    <?php endif; ?>
                </h1>
                <p class="text-muted mb-0">Gerencie todos os seus imóveis</p>
            </div>
            <a href="imovel_criar.php" class="btn btn-success">
                <i class="bi bi-plus me-2"></i>Novo Imóvel
            </a>
        </div>

        <!-- ERRO -->
        <?php if ($erro): ?>
            <div class="alert alert-danger">
                <h5><i class="bi bi-exclamation-triangle me-2"></i>Erro ao carregar imóveis</h5>
                <p class="mb-0"><?php echo htmlspecialchars($erro); ?></p>
            </div>
        <?php endif; ?>

        <!-- BUSCA -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-6">
                        <input type="text" name="search" class="form-control" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Buscar por título ou cidade...">
                    </div>
                    <div class="col-md-3">
                        <select name="status" class="form-select">
                            <option value="todos" <?php echo $status_filter === 'todos' ? 'selected' : ''; ?>>
                                Todos os status
                            </option>
                            <option value="publicado" <?php echo $status_filter === 'publicado' ? 'selected' : ''; ?>>
                                Publicado
                            </option>
                            <option value="rascunho" <?php echo $status_filter === 'rascunho' ? 'selected' : ''; ?>>
                                Rascunho
                            </option>
                            <option value="vendido" <?php echo $status_filter === 'vendido' ? 'selected' : ''; ?>>
                                Vendido
                            </option>
                            <option value="alugado" <?php echo $status_filter === 'alugado' ? 'selected' : ''; ?>>
                                Alugado
                            </option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="bi bi-search me-1"></i>Buscar
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- LISTA DE IMÓVEIS -->
        <?php if (empty($imoveis) && !$erro): ?>
            <div class="card">
                <div class="card-body">
                    <div class="empty-state">
                        <i class="bi bi-house-door"></i>
                        <h3 class="mb-3">Nenhum imóvel encontrado</h3>
                        <p class="text-muted mb-4">
                            <?php if ($search || $status_filter !== 'todos'): ?>
                                Tente ajustar sua busca ou <a href="imoveis.php">ver todos os imóveis</a>.
                            <?php else: ?>
                                Você ainda não cadastrou nenhum imóvel. Comece agora!
                            <?php endif; ?>
                        </p>
                        <a href="imovel_criar.php" class="btn btn-success btn-lg">
                            <i class="bi bi-plus-circle me-2"></i>Cadastrar Primeiro Imóvel
                        </a>
                    </div>
                </div>
            </div>
        <?php elseif (!empty($imoveis)): ?>
            <div class="card">
                <div class="table-responsive">
                    <table class="table table-dark table-hover mb-0">
                        <thead>
                            <tr>
                                <th width="60">ID</th>
                                <th>Título</th>
                                <th>Preço</th>
                                <th>Cidade</th>
                                <th>Status</th>
                                <th>Data</th>
                                <th width="140">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($imoveis as $imovel): ?>
                            <tr>
                                <td>
                                    <span class="badge bg-secondary"><?php echo $imovel['id']; ?></span>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($imovel['titulo']); ?></strong>
                                    <?php if (!empty($imovel['tipo'])): ?>
                                        <br><small class="text-muted"><?php echo htmlspecialchars($imovel['tipo']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="text-success fw-bold">
                                        <?php echo format_money($imovel['preco']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($imovel['cidade']); ?>
                                    <?php if (!empty($imovel['uf'])): ?>
                                        <br><small class="text-muted"><?php echo htmlspecialchars($imovel['uf']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="badge <?php echo get_status_badge_class($imovel['status']); ?>">
                                        <?php echo ucfirst($imovel['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php echo date('d/m/Y', strtotime($imovel['created_at'])); ?>
                                    </small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <a href="imovel_editar.php?id=<?php echo $imovel['id']; ?>" 
                                           class="btn btn-outline-primary" title="Editar">
                                            <i class="bi bi-pencil"></i>
                                        </a>
                                        <button class="btn btn-outline-danger" 
                                                onclick="confirmarExclusao(<?php echo $imovel['id']; ?>, '<?php echo htmlspecialchars($imovel['titulo'], ENT_QUOTES); ?>')" 
                                                title="Excluir">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function confirmarExclusao(id, titulo) {
            if (confirm('Tem certeza que deseja excluir o imóvel "' + titulo + '"?\n\nEsta ação não pode ser desfeita.')) {
                window.location.href = 'imovel_excluir.php?id=' + id;
            }
        }
    </script>
</body>
</html>
