<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('imoveis','view');

$BASE       = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title = 'Imóveis';

$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* ===================== Filtros (GET) ===================== */
$flt_status       = isset($_GET['status']) ? trim((string)$_GET['status']) : '';
$flt_tipo         = isset($_GET['tipo']) ? trim((string)$_GET['tipo']) : '';
$flt_condominioId = isset($_GET['condominio_id']) ? (int)$_GET['condominio_id'] : 0;
$flt_q            = isset($_GET['q']) ? trim((string)$_GET['q']) : '';

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

/* ===================== Opções dos filtros ===================== */
$tipos  = [];
$condos = [];
$erro   = null;
try {
    // Tipos distintos (se a coluna existir)
    try {
        $tipos = fetch_all("SELECT DISTINCT tipo FROM properties WHERE tipo IS NOT NULL AND tipo <> '' ORDER BY tipo ASC LIMIT 200", [], true);
    } catch (Throwable $e) { $tipos = []; }

    // Condomínios (tabela que criamos)
    try {
        $condos = fetch_all("SELECT id, nome FROM condominios ORDER BY nome ASC", [], true);
    } catch (Throwable $e) { $condos = []; }
} catch (Throwable $e) {
    $erro = $e->getMessage();
}

/* ===================== Buscar imóveis ===================== */
$imoveis = [];
try {
    // Tentativa com JOIN (para trazer nome do condomínio)
    $params = [];
    $sql =
      "SELECT p.*, c.nome AS condominio_nome
         FROM properties p
    LEFT JOIN condominios c ON c.id = p.condominio_id
        WHERE 1=1 ";

    if ($flt_status !== '') {
        $sql .= " AND p.status = ? ";
        $params[] = $flt_status;
    }
    if ($flt_tipo !== '') {
        $sql .= " AND p.tipo = ? ";
        $params[] = $flt_tipo;
    }
    if ($flt_condominioId > 0) {
        $sql .= " AND p.condominio_id = ? ";
        $params[] = $flt_condominioId;
    }
    if ($flt_q !== '') {
        $sql .= " AND (p.titulo LIKE ? OR p.cidade LIKE ? OR p.bairro LIKE ?) ";
        $like = "%{$flt_q}%";
        array_push($params, $like, $like, $like);
    }

    $sql .= " ORDER BY p.created_at DESC LIMIT 200";
    $imoveis = fetch_all($sql, $params, true);

} catch (Throwable $e) {
    // Fallback sem JOIN
    try {
        $params = [];
        $sql =
          "SELECT *
             FROM properties p
            WHERE 1=1 ";

        if ($flt_status !== '') {
            $sql .= " AND p.status = ? ";
            $params[] = $flt_status;
        }
        if ($flt_tipo !== '') {
            $sql .= " AND p.tipo = ? ";
            $params[] = $flt_tipo;
        }
        if ($flt_condominioId > 0) {
            $sql .= " AND p.condominio_id = ? ";
            $params[] = $flt_condominioId;
        }
        if ($flt_q !== '') {
            $sql .= " AND (p.titulo LIKE ? OR p.cidade LIKE ? OR p.bairro LIKE ?) ";
            $like = "%{$flt_q}%";
            array_push($params, $like, $like, $like);
        }

        $sql .= " ORDER BY p.created_at DESC LIMIT 200";
        $imoveis = fetch_all($sql, $params, true);
    } catch (Throwable $e2) {
        $erro = $e2->getMessage();
        $imoveis = [];
    }
}

/* ===================== Escopo de VIEW 'own' (filtro em PHP) ===================== */
$scopeView = rbac_scope('imoveis','view'); // 'none' | 'own' | 'all'
if ($scopeView === 'own') {
    $imoveis = array_values(array_filter($imoveis, function($i) use ($me_id) {
        if (isset($i['owner_user_id'])) return ((int)$i['owner_user_id'] === $me_id);
        if (isset($i['created_by']))   return ((int)$i['created_by']   === $me_id);
        return false; // por segurança
    }));
}

$total_imoveis = is_array($imoveis) ? count($imoveis) : 0;

/* ===================== Mensagens pós-ação ===================== */
$msg_ok  = isset($_GET['ok'])      ? (string)$_GET['ok'] : null;
$msg_err = isset($_GET['err'])     ? (string)$_GET['err'] : null;
$deleted = isset($_GET['deleted']) ? (int)$_GET['deleted'] : 0;

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<div class="container mt-4 mb-5">

  <div class="d-flex justify-content-between align-items-center mb-3 flex-wrap gap-2">
    <div>
      <h1 class="mb-1 fw-bold">
        <i class="bi bi-house-door me-2 text-success"></i>
        Imóveis
        <?php if ($total_imoveis > 0): ?>
          <span class="badge bg-secondary"><?= $total_imoveis ?></span>
        <?php endif; ?>
      </h1>
      <p class="text-muted mb-0">Gerencie seus imóveis. Use os filtros para refinar a lista.</p>
    </div>

    <div class="d-flex gap-2">
      <?php if (rbac_can('imoveis','create')): ?>
        <a href="<?= $BASE ?>/imoveis/novo" class="btn btn-success">
          <i class="bi bi-plus me-2"></i>Novo Imóvel
        </a>
      <?php endif; ?>
    </div>
  </div>

  <!-- Mensagens -->
  <?php if ($deleted): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i>Imóvel excluído com sucesso.
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if (!empty($msg_ok)): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i><?= h($msg_ok) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if (!empty($msg_err)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i><?= h($msg_err) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if ($erro): ?>
    <div class="alert alert-danger">
      <h5 class="mb-1"><i class="bi bi-exclamation-triangle me-2"></i>Erro ao carregar imóveis</h5>
      <div class="mb-0"><?= h($erro) ?></div>
    </div>
  <?php endif; ?>

  <!-- Filtros -->
  <div class="card mb-3">
    <div class="card-body">
      <form method="get" class="row g-2 align-items-end">
        <div class="col-md-3">
          <label class="form-label mb-0">Status</label>
          <select name="status" class="form-select">
            <option value="">Todos</option>
            <?php
              $sts = ['publicado'=>'Publicado','rascunho'=>'Rascunho','vendido'=>'Vendido','alugado'=>'Alugado'];
              foreach ($sts as $k=>$lbl):
            ?>
              <option value="<?= h($k) ?>" <?= $flt_status===$k?'selected':'' ?>><?= h($lbl) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-md-3">
          <label class="form-label mb-0">Tipo</label>
          <select name="tipo" class="form-select">
            <option value="">Todos</option>
            <?php foreach ($tipos as $t): $tv = (string)($t['tipo'] ?? ''); if ($tv==='') continue; ?>
              <option value="<?= h($tv) ?>" <?= $flt_tipo===$tv?'selected':'' ?>><?= h($tv) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-md-3">
          <label class="form-label mb-0">Condomínio</label>
          <select name="condominio_id" class="form-select">
            <option value="0">Todos</option>
            <?php foreach ($condos as $c): ?>
              <option value="<?= (int)$c['id'] ?>" <?= $flt_condominioId===(int)$c['id']?'selected':'' ?>>
                <?= h($c['nome']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-md-3">
          <label class="form-label mb-0">Busca</label>
          <input type="text" name="q" class="form-control" placeholder="Título, cidade ou bairro" value="<?= h($flt_q) ?>">
        </div>

        <div class="col-12 d-flex gap-2 mt-2">
          <button class="btn btn-primary"><i class="bi bi-funnel me-1"></i> Filtrar</button>
          <a href="<?= $BASE ?>/imoveis" class="btn btn-outline-secondary">Limpar</a>
        </div>
      </form>
    </div>
  </div>

  <?php if (empty($imoveis) && !$erro): ?>
    <div class="card">
      <div class="card-body text-center py-5">
        <i class="bi bi-house-door display-1 text-muted mb-3"></i>
        <h3 class="mb-3">Nenhum imóvel encontrado</h3>
        <p class="text-muted mb-4">Ajuste os filtros ou cadastre um novo imóvel.</p>
        <?php if (rbac_can('imoveis','create')): ?>
          <a href="<?= $BASE ?>/imoveis/novo" class="btn btn-success btn-lg">
            <i class="bi bi-plus-circle me-2"></i>Novo Imóvel
          </a>
        <?php endif; ?>
      </div>
    </div>
  <?php else: ?>
    <div class="card">
      <div class="table-responsive">
        <table class="table table-dark table-hover mb-0 align-middle">
          <thead>
            <tr>
              <th width="70">ID</th>
              <th>Título</th>
              <th width="160">Preço</th>
              <th width="180">Cidade</th>
              <th width="160">Condomínio</th>
              <th width="140">Status</th>
              <th width="210" class="text-center">Ações</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($imoveis as $imovel): ?>
              <?php
                // RBAC por linha (update/delete conforme escopo)
                $canUpdate = false;
                if (rbac_can('imoveis','update')) {
                    $su = rbac_scope('imoveis','update'); // 'own' | 'all'
                    if ($su === 'all') {
                        $canUpdate = true;
                    } else {
                        $owner = $imovel['owner_user_id'] ?? $imovel['created_by'] ?? null;
                        $canUpdate = ($owner !== null && (int)$owner === $me_id);
                    }
                }

                $canDelete = false;
                if (rbac_can('imoveis','delete')) {
                    $sd = rbac_scope('imoveis','delete'); // 'own' | 'all'
                    if ($sd === 'all') {
                        $canDelete = true;
                    } else {
                        $owner = $imovel['owner_user_id'] ?? $imovel['created_by'] ?? null;
                        $canDelete = ($owner !== null && (int)$owner === $me_id);
                    }
                }

                $status_classes = [
                  'publicado' => 'success',
                  'rascunho'  => 'secondary',
                  'vendido'   => 'danger',
                  'alugado'   => 'info'
                ];
                $st = $status_classes[$imovel['status']] ?? 'secondary';

                $condNome = $imovel['condominio_nome'] ?? '';
              ?>
              <tr>
                <td class="text-center"><?= (int)$imovel['id'] ?></td>
                <td><strong><?= h($imovel['titulo']) ?></strong></td>
                <td><span class="text-success fw-bold"><?= format_money($imovel['preco'] ?? 0) ?></span></td>
                <td><?= h($imovel['cidade'] ?? '') ?></td>
                <td><?= h($condNome) ?></td>
                <td>
                  <span class="badge bg-<?= h($st) ?>">
                    <?= h(ucfirst($imovel['status'] ?? '')) ?>
                  </span>
                </td>
                <td class="text-center">
                  <div class="btn-group btn-group-sm">
                    <!-- visualizar -->
                    <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/visualizar"
                       class="btn btn-outline-success" title="Visualizar imóvel" target="_blank">
                      <i class="bi bi-eye"></i>
                    </a>

                    <?php if ($canUpdate): ?>
                      <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/editar"
                         class="btn btn-outline-primary" title="Editar imóvel">
                        <i class="bi bi-pencil"></i>
                      </a>
                      <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/fotos"
                         class="btn btn-outline-info" title="Gerenciar fotos">
                        <i class="bi bi-images"></i>
                      </a>
                    <?php endif; ?>

                    <!-- Exportar PDF -->
                    <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/exportar"
                       class="btn btn-outline-light" title="Exportar PDF" target="_blank">
                      <i class="bi bi-file-earmark-pdf"></i>
                    </a>

                    <?php if ($canDelete): ?>
                      <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/excluir"
                         class="btn btn-outline-danger" title="Excluir imóvel"
                         onclick="return confirm('Tem certeza que deseja excluir este imóvel?\n\nTodas as fotos e vínculos serão removidos. Esta ação não pode ser desfeita.');">
                        <i class="bi bi-trash"></i>
                      </a>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>

      <?php
        $publicados = count(array_filter($imoveis, fn($i) => ($i['status'] ?? '') === 'publicado'));
        $rascunhos  = count(array_filter($imoveis, fn($i) => ($i['status'] ?? '') === 'rascunho'));
      ?>
      <div class="card-footer text-muted d-flex justify-content-between align-items-center">
        <span><i class="bi bi-info-circle me-2"></i>Total: <strong><?= $total_imoveis ?></strong> imóvel<?= ($total_imoveis!=1?'eis':'') ?></span>
        <div>
          <span class="badge bg-success me-1"><?= $publicados ?> publicado<?= $publicados!=1?'s':'' ?></span>
          <span class="badge bg-secondary"><?= $rascunhos ?> rascunho<?= $rascunhos!=1?'s':'' ?></span>
        </div>
      </div>
    </div>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>