<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../../app/db.php';
require_once __DIR__ . '/../../../app/rbac.php';

rbac_require('admin','access'); // continua exigindo acesso ao módulo Admin

$BASE       = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title = 'Admin • Condomínios';

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

$ok=null; $err=null;

/* Ações POST: criar / ativar / desativar — agora com RBAC por ação */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) throw new Exception('CSRF inválido');

    $action = (string)($_POST['action'] ?? '');
    if ($action === 'add') {
      if (!rbac_can('condominios','create')) throw new Exception('Sem permissão para cadastrar condomínios.');
      $nome   = trim((string)($_POST['nome'] ?? ''));
      $cidade = trim((string)($_POST['cidade'] ?? ''));
      $uf     = strtoupper(trim((string)($_POST['uf'] ?? '')));
      if ($nome === '') throw new Exception('Informe o nome do condomínio.');
      execute("INSERT INTO condominios (nome, cidade, uf, ativo, created_at) VALUES (?, ?, ?, 1, NOW())", [$nome,$cidade,$uf]);
      $ok = 'Condomínio cadastrado.';
    }
    if ($action === 'ativar' || $action === 'desativar') {
      if (!rbac_can('condominios','update')) throw new Exception('Sem permissão para alterar status de condomínios.');
      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) throw new Exception('ID inválido.');
      $val = ($action === 'ativar') ? 1 : 0;
      execute("UPDATE condominios SET ativo=?, updated_at=NOW() WHERE id=?", [$val, $id]);
      $ok = ($val ? 'Ativado' : 'Desativado') . ' com sucesso.';
    }
  } catch (Throwable $e) {
    $err = $e->getMessage();
  }
}

/* Listagem */
$rows = [];
try {
  $rows = fetch_all("SELECT id, nome, cidade, uf, ativo, created_at, updated_at FROM condominios ORDER BY nome ASC", [], true) ?: [];
} catch (Throwable $e) {}

include __DIR__ . '/../../../layout/header.php';
include __DIR__ . '/../../../layout/nav.php';
?>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0"><i class="bi bi-building-add me-2 text-success"></i>Condomínios</h1>
    <a class="btn btn-outline-secondary" href="<?=$BASE?>/admin"><i class="bi bi-arrow-left"></i> Voltar</a>
  </div>

  <?php if ($ok): ?><div class="alert alert-success"><?=$ok?></div><?php endif; ?>
  <?php if ($err): ?><div class="alert alert-danger"><?=$err?></div><?php endif; ?>

  <?php if (rbac_can('condominios','create')): ?>
  <div class="card mb-3">
    <div class="card-header"><i class="bi bi-plus-circle me-1"></i>Novo Condomínio</div>
    <div class="card-body">
      <form method="post" class="row g-2">
        <input type="hidden" name="csrf" value="<?=$CSRF?>">
        <input type="hidden" name="action" value="add">
        <div class="col-md-6">
          <label class="form-label">Nome *</label>
          <input type="text" name="nome" class="form-control" required>
        </div>
        <div class="col-md-4">
          <label class="form-label">Cidade</label>
          <input type="text" name="cidade" class="form-control">
        </div>
        <div class="col-md-2">
          <label class="form-label">UF</label>
          <input type="text" name="uf" class="form-control" maxlength="2" placeholder="GO">
        </div>
        <div class="col-12">
          <button class="btn btn-primary"><i class="bi bi-check2"></i> Cadastrar</button>
        </div>
      </form>
    </div>
  </div>
  <?php endif; ?>

  <div class="card">
    <div class="table-responsive">
      <table class="table table-dark table-hover mb-0 align-middle">
        <thead>
          <tr>
            <th>Nome</th>
            <th style="width:180px">Cidade/UF</th>
            <th style="width:120px">Status</th>
            <th style="width:160px" class="text-end">Ações</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td><strong><?=htmlspecialchars($r['nome'])?></strong></td>
              <td><?= htmlspecialchars(trim(($r['cidade'] ?? '').($r['uf'] ? ' / '.$r['uf'] : ''))) ?: '—' ?></td>
              <td>
                <span class="badge bg-<?= $r['ativo'] ? 'success' : 'secondary' ?>">
                  <?= $r['ativo'] ? 'Ativo' : 'Inativo' ?>
                </span>
              </td>
              <td class="text-end">
                <?php if (rbac_can('condominios','update')): ?>
                  <form method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?=$CSRF?>">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <input type="hidden" name="action" value="<?= $r['ativo']?'desativar':'ativar' ?>">
                    <button class="btn btn-sm btn-outline-<?= $r['ativo']?'warning':'success' ?>" title="<?= $r['ativo']?'Desativar':'Ativar' ?>">
                      <i class="bi bi-<?= $r['ativo']?'pause-circle':'play-circle' ?>"></i>
                    </button>
                  </form>
                <?php else: ?>
                  <button class="btn btn-sm btn-outline-secondary" disabled title="Sem permissão">
                    <i class="bi bi-slash-circle"></i>
                  </button>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; if (!$rows): ?>
            <tr><td colspan="4" class="text-center text-muted py-4">Nenhum condomínio cadastrado.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../../layout/footer.php'; ?>