<?php
/**
 * SISTEMA DE BANCO DE DADOS - TROYA CRM
 * Funções para conexão e manipulação do banco de dados
 */

$config = require __DIR__ . "/../config/config.php";

// Define BASE_URL caso não esteja definida
if (!defined('BASE_URL') && isset($config['app']['base_url_admin'])) {
    define('BASE_URL', $config['app']['base_url_admin']);
} elseif (!defined('BASE_URL')) {
    define('BASE_URL', 'https://ztr.com.br/troyacrm');
}

/**
 * Retorna conexão PDO com o banco de dados
 */
function db($readonly = false) {
    static $pdo = null;
    static $pdo_ro = null;
    static $config_cache = null;

    if ($config_cache === null) {
        $config_cache = require __DIR__ . "/../config/config.php";
    }

    if ($readonly && $pdo_ro !== null) return $pdo_ro;
    if (!$readonly && $pdo !== null)   return $pdo;

    $db_config = $readonly ? ($config_cache["db_readonly"] ?? $config_cache["db"]) : $config_cache["db"];

    try {
        $host    = $db_config["host"];
        $port    = !empty($db_config["port"]) ? ";port={$db_config["port"]}" : "";
        $dbname  = $db_config["name"];
        $charset = $db_config["charset"] ?? "utf8mb4";
        $dsn     = "mysql:host={$host}{$port};dbname={$dbname};charset={$charset}";

        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$charset}"
        ];

        $connection = new PDO($dsn, $db_config["user"], $db_config["pass"], $options);

        if ($readonly) $pdo_ro = $connection;
        else           $pdo    = $connection;

        return $connection;

    } catch (PDOException $e) {
        error_log("Troya CRM - DB Connection Error: " . $e->getMessage());
        throw new Exception("Erro de conexão com banco de dados: " . $e->getMessage());
    }
}

/**
 * Executa uma query SQL com prepared statements
 */
function query($sql, $params = [], $readonly = false) {
    try {
        $pdo = db($readonly);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params ?: []);
        return $stmt;
    } catch (PDOException $e) {
        error_log("Troya CRM - Query Error: " . $e->getMessage() . " | SQL: " . $sql);
        throw new Exception("Erro ao executar consulta: " . $e->getMessage());
    }
}

/**
 * Busca um único registro
 */
function fetch_one($sql, $params = [], $readonly = false) {
    $stmt = query($sql, $params, $readonly);
    return $stmt->fetch();
}

/**
 * Busca múltiplos registros
 */
function fetch_all($sql, $params = [], $readonly = false) {
    $stmt = query($sql, $params, $readonly);
    return $stmt->fetchAll();
}

/**
 * Executa INSERT/UPDATE/DELETE
 */
function execute($sql, $params = [], $debug = false) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare($sql);
        $ok = $stmt->execute($params);
        if ($debug) {
            echo "<pre>SQL: $sql\nParams: " . print_r($params, true) . "\nResult: " . ($ok ? 'SUCCESS' : 'FAILED') . "</pre>";
        }
        return $ok;
    } catch (PDOException $e) {
        if ($debug) echo "<pre>ERRO: " . $e->getMessage() . "</pre>";
        error_log("Troya CRM - Execute Error: " . $e->getMessage() . " | SQL: " . $sql);
        throw $e;
    }
}

/** Retorna o último ID inserido */
function last_insert_id() { return db()->lastInsertId(); }

/** Transações */
function begin_transaction() { return db()->beginTransaction(); }
function commit()           { return db()->commit(); }
function rollback()         { return db()->rollBack(); }

/** Helpers específicos usados pelo sistema (mantidos aqui por legado) */
function count_properties($status = null) {
    try {
        $sql = "SELECT COUNT(*) as total FROM properties";
        $params = [];
        if ($status) { $sql .= " WHERE status = ?"; $params[] = $status; }
        $r = fetch_one($sql, $params, true);
        return $r ? (int)$r['total'] : 0;
    } catch (Exception $e) { error_log("count_properties error: " . $e->getMessage()); return 0; }
}

function count_clients() {
    try {
        $r = fetch_one("SELECT COUNT(*) as total FROM clients", [], true);
        return $r ? (int)$r['total'] : 0;
    } catch (Exception $e) { error_log("count_clients error: " . $e->getMessage()); return 0; }
}

function get_recent_properties($limit = 5) {
    try {
        $sql = "SELECT id, titulo, preco, cidade, status, created_at
                FROM properties
                ORDER BY created_at DESC
                LIMIT ?";
        return fetch_all($sql, [(int)$limit], true);
    } catch (Exception $e) { error_log("get_recent_properties error: " . $e->getMessage()); return []; }
}

function get_recent_clients($limit = 5) {
    try {
        $limit = max(1, (int)$limit);
        $sql = "SELECT id, nome_completo AS nome, telefone_principal AS whatsapp, email, created_at
                FROM clients
                ORDER BY created_at DESC, id DESC
                LIMIT {$limit}";
        return fetch_all($sql, [], true);
    } catch (Exception $e) { error_log("get_recent_clients error: " . $e->getMessage()); return []; }
}

/** Utils diversos */
function format_money($value) {
    return "R$ " . number_format((float)($value ?: 0), 2, ",", ".");
}
function parse_money($value) {
    if (empty($value)) return 0.0;
    $value = preg_replace('/[^\d,.]/', '', $value);
    $value = str_replace(',', '.', $value);
    $parts = explode('.', $value);
    if (count($parts) > 2) $value = implode('', array_slice($parts, 0, -1)) . '.' . end($parts);
    return (float)$value;
}
function sanitize($data) { return $data === null ? "" : htmlspecialchars((string)$data, ENT_QUOTES, "UTF-8"); }
function generate_slug($text) {
    if (empty($text)) return 'imovel-' . time();
    $text = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $text);
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9]+/', '-', $text);
    $text = preg_replace('/-+/', '-', $text);
    $text = trim($text, '-');
    $text = substr($text, 0, 200);
    return $text . '-' . time();
}
function format_cpf($cpf) {
    if (empty($cpf)) return null;
    $cpf = preg_replace('/\D/', '', $cpf);
    if (strlen($cpf) !== 11) return null;
    return substr($cpf,0,3).'.'.substr($cpf,3,3).'.'.substr($cpf,6,3).'-'.substr($cpf,9,2);
}
function format_phone($phone) {
    if (empty($phone)) return null;
    $phone = preg_replace('/\D/', '', $phone);
    if (strlen($phone) < 10 || strlen($phone) > 11) return $phone;
    if (strlen($phone) === 10) {
        return '(' . substr($phone,0,2) . ') ' . substr($phone,2,4) . '-' . substr($phone,6,4);
    }
    return '(' . substr($phone,0,2) . ') ' . substr($phone,2,5) . '-' . substr($phone,7,4);
}
function is_valid_email($email) { return filter_var($email, FILTER_VALIDATE_EMAIL) !== false; }
function hash_password($password) { return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]); }
function verify_password($password, $hash) { return password_verify($password, $hash); }
function format_date($date, $format = 'd/m/Y') {
    if (empty($date)) return '';
    try { return (new DateTime($date))->format($format); } catch (Exception $e) { return $date; }
}
function format_datetime($datetime, $format = 'd/m/Y H:i') { return format_date($datetime, $format); }
function time_ago($datetime) {
    if (empty($datetime)) return '';
    try {
        $now = new DateTime(); $ago = new DateTime($datetime); $diff = $now->diff($ago);
        if ($diff->y>0) return $diff->y.' ano'.($diff->y>1?'s':'').' atrás';
        if ($diff->m>0) return $diff->m.' mês'.($diff->m>1?'es':'').' atrás';
        if ($diff->d>0) return $diff->d.' dia'.($diff->d>1?'s':'').' atrás';
        if ($diff->h>0) return $diff->h.' hora'.($diff->h>1?'s':'').' atrás';
        if ($diff->i>0) return $diff->i.' minuto'.($diff->i>1?'s':'').' atrás';
        return 'agora mesmo';
    } catch (Exception $e) { return $datetime; }
}
function truncate($text, $length = 100, $suffix = '...') {
    if (empty($text)) return '';
    return mb_strlen($text) <= $length ? $text : (mb_substr($text, 0, $length) . $suffix);
}
function build_query_string($params) { return http_build_query($params); }