<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Editar Atendimento';
$active     = 'atendimentos';

rbac_require('atendimentos','update');

$BASE  = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* Status válidos */
$STATUS = ['novo','em_atendimento','negociacao','finalizado'];

/* Helpers */
if (!function_exists('format_date')) {
  function format_date($dt,$fmt='Y-m-d\TH:i'){ return $dt ? date($fmt, strtotime($dt)) : ''; }
}
if (!function_exists('format_money')) {
  function format_money($v) { return 'R$ ' . number_format((float)$v, 2, ',', '.'); }
}

/* Quem pode trocar o responsável? */
$canAssignOwner = (rbac_scope('atendimentos','update') === 'all') || rbac_can('admin','access');

/* Checa se existe coluna property_id */
$hasPropertyCol = false;
try { fetch_one("SELECT property_id FROM interactions LIMIT 1", [], true); $hasPropertyCol = true; } catch (Throwable $e) {}

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) { header("Location: {$BASE}/atendimentos"); exit; }

$erro = null; $ok = null;

/* Carrega registro */
try {
  $row = fetch_one("
    SELECT i.*, c.nome_completo AS cliente_nome, u.nome AS owner_nome
      FROM interactions i
 LEFT JOIN clients c ON c.id = i.client_id
 LEFT JOIN users   u ON u.id = i.owner_user_id
     WHERE i.id = ?
  ", [$id], true);

  if (!$row) { throw new Exception('Atendimento não encontrado.'); }

  // Enforce escopo 'own', se aplicável
  $scopeU = rbac_scope('atendimentos','update'); // 'own' | 'all'
  if ($scopeU === 'own') {
    $owner   = (int)($row['owner_user_id'] ?? 0);
    $creator = (int)($row['created_by'] ?? 0);
    if (!in_array($me_id, [$owner, $creator], true)) {
      http_response_code(403); exit('Sem permissão.');
    }
  }
} catch (Throwable $e) {
  $erro = $e->getMessage();
  $row = null;
}

/* Carrega combos quando necessário */
$owners   = [];
$imoveis  = [];

if ($canAssignOwner) {
  try { $owners = fetch_all("SELECT id, nome FROM users ORDER BY nome ASC", [], true) ?: []; } catch (Throwable $e) {}
}
if ($hasPropertyCol) {
  try {
    $imoveis = fetch_all("
      SELECT id, titulo, cidade, uf, tipo, finalidade, preco
        FROM properties
    ORDER BY updated_at DESC, id DESC LIMIT 300
    ", [], true) ?: [];
  } catch (Throwable $e) {}
}

/* POST: salvar */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $row) {
  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) throw new Exception('CSRF inválido.');

    $status      = (string)($_POST['status'] ?? 'novo');
    $nota        = trim($_POST['nota'] ?? '');
    $next_at     = trim($_POST['next_followup_at'] ?? '');
    $owner_id_in = (int)($_POST['owner_user_id'] ?? 0);
    $property_id = (int)($_POST['property_id'] ?? 0);

    if (!in_array($status, $STATUS, true)) throw new Exception('Status inválido.');

    $next_dt = $next_at ? date('Y-m-d H:i:s', strtotime($next_at)) : null;

    // Estado ANTES da alteração (para logs)
    $old_owner_id   = (int)($row['owner_user_id'] ?? 0);
    $old_owner_name = (string)($row['owner_nome'] ?? '—');
    $old_prop_id    = (int)($row['property_id'] ?? 0);

    // Se pode trocar responsável, valida
    $owner_id = $old_owner_id;
    if ($canAssignOwner) {
      if ($owner_id_in <= 0) throw new Exception('Selecione o responsável.');
      $chk = fetch_one("SELECT id, nome FROM users WHERE id = ?", [$owner_id_in], true);
      if (!$chk) throw new Exception('Responsável inválido.');
      $owner_id = (int)$chk['id'];
      $new_owner_name = (string)$chk['nome'];
    } else {
      // mantém o atual
      $new_owner_name = $old_owner_name;
    }

    // Valida imóvel se existir col e foi informado
    $new_prop_id = $old_prop_id;
    if ($hasPropertyCol) {
      if ($property_id > 0) {
        $pp = fetch_one("SELECT id FROM properties WHERE id = ?", [$property_id], true);
        if (!$pp) throw new Exception('Imóvel inválido.');
        $new_prop_id = (int)$property_id;
      } else {
        $new_prop_id = 0; // removendo vínculo
      }
    }

    // Monta SQL dinamicamente
    $sets = "status = ?, nota = ?, next_followup_at = ?, updated_at = NOW()";
    $args = [$status, ($nota !== '' ? $nota : null), $next_dt];

    if ($canAssignOwner) { $sets .= ", owner_user_id = ?"; $args[] = $owner_id; }
    if ($hasPropertyCol) { $sets .= ", property_id = ?";   $args[] = ($new_prop_id ?: null); }

    $args[] = $id;

    // Transação para atualizar e registrar logs
    $pdo = db();
    $pdo->beginTransaction();

    execute("UPDATE interactions SET {$sets} WHERE id = ?", $args);

    // LOGS: responsável alterado
    if ($canAssignOwner && $owner_id !== $old_owner_id) {
      $notaLog = sprintf("Responsável alterado de \"%s\" (ID %d) para \"%s\" (ID %d).",
                         $old_owner_name ?: '—', $old_owner_id ?: 0,
                         $new_owner_name ?: '—', $owner_id ?: 0);
      execute("
        INSERT INTO interaction_logs (interaction_id, client_id, status, nota, next_followup_at, created_by, created_at)
        VALUES (?, ?, ?, ?, NULL, ?, NOW())
      ", [$id, (int)$row['client_id'], $status, $notaLog, $me_id ?: null]);
    }

    // LOGS: imóvel alterado (se a coluna existir)
    if ($hasPropertyCol && $new_prop_id !== $old_prop_id) {
      $txt = ($new_prop_id && $old_prop_id)
             ? "Imóvel associado alterado de #{$old_prop_id} para #{$new_prop_id}."
             : ($new_prop_id ? "Imóvel associado definido para #{$new_prop_id}." : "Imóvel associado removido.");
      execute("
        INSERT INTO interaction_logs (interaction_id, client_id, status, nota, next_followup_at, created_by, created_at)
        VALUES (?, ?, ?, ?, NULL, ?, NOW())
      ", [$id, (int)$row['client_id'], $status, $txt, $me_id ?: null]);
    }

    $pdo->commit();

    $ok = 'Atendimento atualizado com sucesso.';

    // Recarrega o registro já com alterações
    $row = fetch_one("
      SELECT i.*, c.nome_completo AS cliente_nome, u.nome AS owner_nome
        FROM interactions i
   LEFT JOIN clients c ON c.id = i.client_id
   LEFT JOIN users   u ON u.id = i.owner_user_id
       WHERE i.id = ?
    ", [$id], true);

  } catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) { $pdo->rollBack(); }
    $erro = $e->getMessage();
  }
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0"><i class="bi bi-pencil-square text-primary"></i> Editar Atendimento #<?= (int)$id ?></h1>
    <a href="<?=$BASE?>/atendimentos/visualizar?id=<?= (int)$id ?>" class="btn btn-outline-secondary"><i class="bi bi-eye"></i> Ver</a>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success alert-dismissible fade show"><i class="bi bi-check2-circle me-2"></i><?=htmlspecialchars($ok)?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
  <?php endif; ?>
  <?php if ($erro): ?>
    <div class="alert alert-danger alert-dismissible fade show"><i class="bi bi-exclamation-triangle me-2"></i><?=htmlspecialchars($erro)?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
  <?php endif; ?>

  <?php if ($row): ?>
  <form class="card card-body" method="post">
    <input type="hidden" name="csrf" value="<?=htmlspecialchars($CSRF)?>">
    <div class="row g-3">
      <div class="col-md-6">
        <label class="form-label">Cliente</label>
        <div class="form-control-plaintext fw-semibold">
          <a class="link-light" href="<?=$BASE?>/clientes/visualizar?id=<?=(int)$row['client_id']?>"><?=htmlspecialchars($row['cliente_nome'] ?? '—')?></a>
        </div>
      </div>

      <?php if ($canAssignOwner): ?>
      <div class="col-md-6">
        <label class="form-label">Responsável *</label>
        <select name="owner_user_id" class="form-select" required>
          <option value="">Selecione...</option>
          <?php foreach ($owners as $o): ?>
            <option value="<?=$o['id']?>" <?= ((int)$row['owner_user_id']===$o['id'])?'selected':'' ?>>
              <?= htmlspecialchars($o['nome']) ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>
      <?php else: ?>
      <div class="col-md-6">
        <label class="form-label">Responsável</label>
        <div class="form-control-plaintext"><?=htmlspecialchars($row['owner_nome'] ?? '—')?></div>
      </div>
      <?php endif; ?>

      <?php if ($hasPropertyCol): ?>
      <div class="col-md-12">
        <label class="form-label">Imóvel (opcional)</label>
        <select name="property_id" class="form-select">
          <option value="">— (nenhum) —</option>
          <?php foreach ($imoveis as $p):
            $label = sprintf("#%d · %s — %s/%s · %s/%s · %s",
              (int)$p['id'], (string)$p['titulo'], (string)$p['cidade'], strtoupper((string)$p['uf']),
              strtoupper((string)$p['tipo']), strtoupper((string)$p['finalidade']), format_money((float)$p['preco'])
            );
          ?>
            <option value="<?=$p['id']?>" <?= ((int)($row['property_id'] ?? 0) === (int)$p['id'])?'selected':'' ?>>
              <?= htmlspecialchars($label) ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>
      <?php endif; ?>

      <div class="col-md-3">
        <label class="form-label">Status *</label>
        <select name="status" class="form-select" required>
          <?php
            $labels = ['novo'=>'Novo','em_atendimento'=>'Em Atendimento','negociacao'=>'Negociação','finalizado'=>'Finalizado'];
            foreach ($STATUS as $st): ?>
            <option value="<?=$st?>" <?=($row['status']===$st)?'selected':''?>><?=$labels[$st]?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label">Próxima ação (data/hora)</label>
        <input type="datetime-local" name="next_followup_at" class="form-control"
               value="<?= htmlspecialchars(format_date($row['next_followup_at'] ?? null)) ?>">
      </div>

      <div class="col-12">
        <label class="form-label">Nota</label>
        <textarea name="nota" class="form-control" rows="4"><?= htmlspecialchars($row['nota'] ?? '') ?></textarea>
      </div>
    </div>

    <div class="d-flex justify-content-end gap-2 mt-4">
      <a class="btn btn-outline-secondary" href="<?=$BASE?>/atendimentos/visualizar?id=<?=(int)$id?>"><i class="bi bi-x-circle"></i> Cancelar</a>
      <button class="btn btn-primary"><i class="bi bi-check-circle"></i> Salvar</button>
    </div>
  </form>
  <?php endif; ?>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>