<?php
/**
 * RBAC minimalista com checkboxes por usuário
 * - Catálogo: permissions (module, action, has_scope)
 * - Overrides: user_permissions (user_id, permission_id, allow, scope)
 *
 * Requisitos de sessão:
 *   $_SESSION['user']['id']
 *   $_SESSION['user']['role']  (se 'admin', tem super-permissão)
 */

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
require_once __DIR__ . '/db.php';

$__pdo = function_exists('db') ? db() : (isset($pdo) ? $pdo : (isset($GLOBALS['pdo']) ? $GLOBALS['pdo'] : null));
if (!($__pdo instanceof PDO)) { throw new RuntimeException('Conexão PDO não encontrada em RBAC'); }

/** Carrega permissões do usuário para a sessão (cache leve) */
function rbac_reload_permissions($userId) {
  global $__pdo;
  $stmt = $__pdo->prepare("
    SELECT p.module, p.action, p.has_scope, up.allow, up.scope
      FROM permissions p
 LEFT JOIN user_permissions up
        ON up.permission_id = p.id AND up.user_id = ?
     ORDER BY p.module, p.action
  ");
  $stmt->execute([$userId]);
  $acl = [];
  while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $m = $r['module']; $a = $r['action'];
    $acl[$m][$a] = [
      'has_scope' => (int)$r['has_scope'],
      'allow'     => is_null($r['allow']) ? null : (int)$r['allow'],
      'scope'     => $r['scope'] ?? 'all', // default 'all' se não setado
    ];
  }
  $_SESSION['acl'] = $acl;
  return $acl;
}

function rbac_user() {
  return $_SESSION['user'] ?? null;
}

function rbac_is_admin() {
  $u = rbac_user();
  return $u && isset($u['role']) && $u['role'] === 'admin';
}

/** Retorna estrutura ACL do usuário (carrega se necessário) */
function rbac_acl() {
  $u = rbac_user();
  if (!$u) return [];
  if (!isset($_SESSION['acl'])) {
    rbac_reload_permissions((int)$u['id']);
  }
  return $_SESSION['acl'] ?? [];
}

/** Checa permissão booleana (sem considerar SQL de escopo) */
function rbac_can($module, $action) {
  if (rbac_is_admin()) return true;
  $acl = rbac_acl();
  if (!isset($acl[$module][$action])) {
    // Sem override: por padrão NEGAR
    return false;
  }
  $node = $acl[$module][$action];
  if ($node['allow'] === null) {
    // Sem override explícito -> negar por padrão
    return false;
  }
  return (bool)$node['allow'];
}

/**
 * Retorna o escopo efetivo para module.action:
 * - 'all' | 'own' | 'none'
 */
function rbac_scope($module, $action) {
  if (rbac_is_admin()) return 'all';
  $acl = rbac_acl();
  if (!isset($acl[$module][$action])) return 'none';
  $node = $acl[$module][$action];
  if ((int)$node['has_scope'] !== 1) return ($node['allow'] ? 'all' : 'none');
  if ($node['allow'] === 0) return 'none';
  $scope = $node['scope'] ?? 'all';
  return in_array($scope, ['own','all'], true) ? $scope : 'none';
}

/**
 * Gera trecho SQL para filtrar por dono quando escopo='own'
 * $alias = apelido da tabela (ex.: 'c' para clients)
 */
function rbac_scope_sql($module, $action, $alias = '') {
  $scope = rbac_scope($module, $action);
  if ($scope !== 'own') return ''; // 'all' ou 'none' não restringe aqui (o 'none' deve ser barrado antes)
  $u = rbac_user();
  $fld = null;
  switch ($module) {
    case 'clientes':      $fld = 'owner_user_id'; break;
    case 'agenda':        $fld = 'user_id';       break;
    case 'atendimentos':  $fld = 'owner_user_id'; break; // <<< NOVO: atende interactions
    // futuros módulos...
  }
  if (!$fld || !$u) return '';
  $a = $alias ? ($alias . '.') : '';
  return " AND {$a}{$fld} = " . (int)$u['id'] . " ";
}

/** Enforce server-side (403) */
function rbac_require($module, $action, $opts = []) {
  if (!rbac_can($module, $action)) {
    http_response_code(403);
    $msg = $opts['message'] ?? 'Você não tem permissão para acessar este módulo/ação.';
    // Tenta incluir página 403 do layout, se existir
    $tpl = __DIR__ . '/../layout/403.php';
    if (file_exists($tpl)) {
      $page_title = 'Acesso negado';
      include $tpl;
    } else {
      echo "<div style='padding:2rem;color:#e6edf3;background:#0d1117;font-family:system-ui'>
              <h3 style='margin:0 0 .5rem'>Acesso negado (403)</h3>
              <div style='opacity:.8'>{$msg}</div>
            </div>";
    }
    exit;
  }
}

/* =========================
   Helpers opcionais p/ Timeline
   ========================= */

/**
 * Escopo efetivo para adicionar timeline em atendimentos:
 * - tenta 'timeline_add' explícito;
 * - se não houver, cai para 'update';
 * - se ainda 'none', cai para 'view' (para permitir leitor lançar timeline no que é dele).
 */
function rbac_scope_timeline_atendimentos() {
  if (rbac_is_admin()) return 'all';
  $acl = rbac_acl();
  // 1) explícito
  if (isset($acl['atendimentos']['timeline_add'])) {
    return rbac_scope('atendimentos', 'timeline_add');
  }
  // 2) fallback update
  $upd = rbac_scope('atendimentos', 'update');
  if ($upd !== 'none') return $upd;
  // 3) fallback view
  return rbac_scope('atendimentos', 'view');
}

/** Verifica se o usuário pode adicionar timeline neste registro (row de interactions) */
function rbac_can_timeline_for_atendimento($row) {
  if (rbac_is_admin()) return true;
  $scope = rbac_scope_timeline_atendimentos(); // 'all' | 'own' | 'none'
  if ($scope === 'all') return true;
  if ($scope === 'own') {
    $me = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
    $owner   = (int)($row['owner_user_id'] ?? 0);
    $creator = (int)($row['created_by'] ?? 0);
    return in_array($me, [$owner, $creator], true);
  }
  return false;
}

/* ==========================================================
   RÓTULOS AMIGÁVEIS (labels) PARA MÓDULOS E AÇÕES DO RBAC
   ========================================================== */

/** Mapa de labels por módulo/ação (adicione/edite conforme necessário) */
$RBAC_LABELS = [
  'admin' => [
    'access'         => 'Acessar Administração',
    'config_update'  => 'Alterar configurações',
    'transfer_owner' => 'Transferir responsável',
  ],
  'agenda' => [
    'view'   => 'Ver agenda',
    'create' => 'Criar compromissos',
    'update' => 'Editar compromissos',
    'delete' => 'Excluir compromissos',
  ],
  'imoveis' => [
    'view'   => 'Ver imóveis',
    'create' => 'Cadastrar imóveis',
    'update' => 'Editar imóveis',
    'delete' => 'Excluir imóveis',
  ],
  'clientes' => [
    'view'   => 'Ver clientes',
    'create' => 'Cadastrar clientes',
    'update' => 'Editar clientes',
    'delete' => 'Excluir clientes',
  ],
  'atendimentos' => [
    'view'        => 'Ver atendimentos',
    'create'      => 'Criar atendimentos',
    'update'      => 'Editar atendimentos',
    'delete'      => 'Excluir atendimentos',
    'timeline_add'=> 'Adicionar timeline',
  ],
  'negocios' => [
    'view'   => 'Ver negócios',
    'create' => 'Criar negócios',
    'update' => 'Editar negócios',
    'delete' => 'Excluir negócios',
  ],
  'usuarios' => [
    'view'   => 'Ver usuários',
    'create' => 'Criar usuários',
    'update' => 'Editar usuários',
    'delete' => 'Excluir usuários',
  ],
];

/** Labels de módulos (se quiser exibir o nome do módulo “bonito”) */
$RBAC_MODULE_LABELS = [
  'admin'         => 'Administração',
  'agenda'        => 'Agenda',
  'imoveis'       => 'Imóveis',
  'clientes'      => 'Clientes',
  'atendimentos'  => 'Atendimentos',
  'negocios'      => 'Negócios',
  'usuarios'      => 'Usuários',
];

/** Retorna o rótulo amigável da ação; fallback formata a chave */
if (!function_exists('rbac_action_label')) {
  function rbac_action_label(string $module, string $action): string {
    global $RBAC_LABELS;
    $m = strtolower($module);
    if (isset($RBAC_LABELS[$m][$action])) {
      return $RBAC_LABELS[$m][$action];
    }
    // fallback: "config_update" -> "Config Update"
    return ucwords(str_replace('_',' ', $action));
  }
}

/** (Opcional) Retorna o rótulo amigável do módulo; fallback capitaliza */
if (!function_exists('rbac_module_label')) {
  function rbac_module_label(string $module): string {
    global $RBAC_MODULE_LABELS;
    $m = strtolower($module);
    return $RBAC_MODULE_LABELS[$m] ?? ucwords(str_replace('_',' ', $module));
  }
}