<?php
/**
 * SISTEMA DE BANCO DE DADOS - TROYA CRM
 * Funções para conexão e manipulação do banco de dados
 */

// Carregar configurações e definir BASE_URL
$config = require __DIR__ . "/../config/config.php";

// Definir BASE_URL se não foi definida
if (!defined('BASE_URL') && isset($config['app']['base_url_admin'])) {
    define('BASE_URL', $config['app']['base_url_admin']);
} elseif (!defined('BASE_URL')) {
    define('BASE_URL', 'https://ztr.com.br/troyacrm');
}

/**
 * Retorna conexão PDO com o banco de dados
 */
function db($readonly = false) {
    static $pdo = null;
    static $pdo_ro = null;
    static $config = null;
    
    // Carregar config apenas uma vez
    if ($config === null) {
        $config = require __DIR__ . "/../config/config.php";
    }
    
    // Retornar conexão existente se já foi criada
    if ($readonly && $pdo_ro !== null) {
        return $pdo_ro;
    }
    if (!$readonly && $pdo !== null) {
        return $pdo;
    }
    
    // Carregar configuração do banco
    $db_config = $readonly ? ($config["db_readonly"] ?? $config["db"]) : $config["db"];
    
    try {
        // Construir DSN
        $host = $db_config["host"];
        $port = !empty($db_config["port"]) ? ";port={$db_config["port"]}" : "";
        $dbname = $db_config["name"];
        $charset = $db_config["charset"] ?? "utf8mb4";
        
        $dsn = "mysql:host={$host}{$port};dbname={$dbname};charset={$charset}";
        
        // Opções do PDO
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$charset}"
        ];
        
        // Criar conexão
        $connection = new PDO($dsn, $db_config["user"], $db_config["pass"], $options);
        
        // Armazenar conexão
        if ($readonly) {
            $pdo_ro = $connection;
        } else {
            $pdo = $connection;
        }
        
        return $connection;
        
    } catch (PDOException $e) {
        error_log("Troya CRM - DB Connection Error: " . $e->getMessage());
        throw new Exception("Erro de conexão com banco de dados: " . $e->getMessage());
    }
}

/**
 * Executa uma query SQL com prepared statements
 */
function query($sql, $params = [], $readonly = false) {
    try {
        $pdo = db($readonly);
        $stmt = $pdo->prepare($sql);
        
        if (!empty($params)) {
            $stmt->execute($params);
        } else {
            $stmt->execute();
        }
        
        return $stmt;
        
    } catch (PDOException $e) {
        error_log("Troya CRM - Query Error: " . $e->getMessage() . " | SQL: " . $sql);
        throw new Exception("Erro ao executar consulta: " . $e->getMessage());
    }
}

/**
 * Busca um único registro
 */
function fetch_one($sql, $params = [], $readonly = false) {
    $stmt = query($sql, $params, $readonly);
    return $stmt->fetch();
}

/**
 * Busca múltiplos registros
 */
function fetch_all($sql, $params = [], $readonly = false) {
    $stmt = query($sql, $params, $readonly);
    return $stmt->fetchAll();
}

/**
 * Executa um comando SQL (INSERT, UPDATE, DELETE)
 * Retorna TRUE em sucesso, FALSE em falha
 */
function execute($sql, $params = [], $debug = false) {
    try {
        $pdo = db(); // Usar função db() em vez de global $pdo
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        if ($debug) {
            echo "<pre>SQL: $sql\nParams: " . print_r($params, true) . "\nResult: " . ($result ? 'SUCCESS' : 'FAILED') . "</pre>";
        }
        
        return $result; // Retorna TRUE se sucesso, FALSE se falhou
        
    } catch (PDOException $e) {
        if ($debug) {
            echo "<pre>ERRO: " . $e->getMessage() . "</pre>";
        }
        error_log("Troya CRM - Execute Error: " . $e->getMessage() . " | SQL: " . $sql);
        throw $e;
    }
}

/**
 * Retorna o último ID inserido
 */
function last_insert_id() {
    return db()->lastInsertId();
}

/**
 * Inicia uma transação
 */
function begin_transaction() {
    return db()->beginTransaction();
}

/**
 * Confirma uma transação
 */
function commit() {
    return db()->commit();
}

/**
 * Reverte uma transação
 */
function rollback() {
    return db()->rollBack();
}

/**
 * Conta imóveis por status
 */
function count_properties($status = null) {
    try {
        $sql = "SELECT COUNT(*) as total FROM properties";
        $params = [];
        
        if ($status) {
            $sql .= " WHERE status = ?";
            $params[] = $status;
        }
        
        $result = fetch_one($sql, $params, true);
        return $result ? (int)$result['total'] : 0;
        
    } catch (Exception $e) {
        error_log("count_properties error: " . $e->getMessage());
        return 0;
    }
}

/**
 * Conta total de clientes
 */
function count_clients() {
    try {
        $result = fetch_one("SELECT COUNT(*) as total FROM clients", [], true);
        return $result ? (int)$result['total'] : 0;
    } catch (Exception $e) {
        error_log("count_clients error: " . $e->getMessage());
        return 0;
    }
}

/**
 * Busca imóveis recentes
 */
function get_recent_properties($limit = 5) {
    try {
        $sql = "SELECT id, titulo, preco, cidade, status, created_at 
                FROM properties 
                ORDER BY created_at DESC 
                LIMIT ?";
        return fetch_all($sql, [(int)$limit], true);
    } catch (Exception $e) {
        error_log("get_recent_properties error: " . $e->getMessage());
        return [];
    }
}

/**
 * Busca clientes recentes
 * (ajuste mínimo: usa nome_completo como 'nome' e evita placeholder no LIMIT)
 */
function get_recent_clients($limit = 5) {
    try {
        $limit = max(1, (int)$limit);
        $sql = "SELECT
                    id,
                    nome_completo AS nome,
                    telefone_principal AS whatsapp,
                    email,
                    created_at
                FROM clients
                ORDER BY created_at DESC, id DESC
                LIMIT {$limit}";
        return fetch_all($sql, [], true);
    } catch (Exception $e) {
        error_log("get_recent_clients error: " . $e->getMessage());
        return [];
    }
}

/**
 * Formata valor monetário para exibição
 */
function format_money($value) {
    if (empty($value)) {
        return "R$ 0,00";
    }
    return "R$ " . number_format((float)$value, 2, ",", ".");
}

/**
 * Converte valor monetário do formulário para float
 */
function parse_money($value) {
    if (empty($value)) {
        return 0.0;
    }
    
    $value = preg_replace('/[^\d,.]/', '', $value);
    $value = str_replace(',', '.', $value);
    
    $parts = explode('.', $value);
    if (count($parts) > 2) {
        $value = implode('', array_slice($parts, 0, -1)) . '.' . end($parts);
    }
    
    return (float)$value;
}

/**
 * Sanitiza string para exibição HTML
 */
function sanitize($data) {
    if ($data === null) {
        return "";
    }
    return htmlspecialchars((string)$data, ENT_QUOTES, "UTF-8");
}

/**
 * Gera slug único a partir de texto
 */
function generate_slug($text) {
    if (empty($text)) {
        return 'imovel-' . time();
    }
    
    $text = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $text);
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9]+/', '-', $text);
    $text = preg_replace('/-+/', '-', $text);
    $text = trim($text, '-');
    $text = substr($text, 0, 200);
    
    return $text . '-' . time();
}

/**
 * Valida e formata CPF
 */
function format_cpf($cpf) {
    if (empty($cpf)) {
        return null;
    }
    
    $cpf = preg_replace('/\D/', '', $cpf);
    
    if (strlen($cpf) !== 11) {
        return null;
    }
    
    return substr($cpf, 0, 3) . '.' . 
           substr($cpf, 3, 3) . '.' . 
           substr($cpf, 6, 3) . '-' . 
           substr($cpf, 9, 2);
}

/**
 * Valida e formata telefone/WhatsApp
 */
function format_phone($phone) {
    if (empty($phone)) {
        return null;
    }
    
    $phone = preg_replace('/\D/', '', $phone);
    
    if (strlen($phone) < 10 || strlen($phone) > 11) {
        return $phone;
    }
    
    if (strlen($phone) === 10) {
        return '(' . substr($phone, 0, 2) . ') ' . 
               substr($phone, 2, 4) . '-' . 
               substr($phone, 6, 4);
    } else {
        return '(' . substr($phone, 0, 2) . ') ' . 
               substr($phone, 2, 5) . '-' . 
               substr($phone, 7, 4);
    }
}

/**
 * Valida email
 */
function is_valid_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Gera hash de senha seguro
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
}

/**
 * Verifica senha contra hash
 */
function verify_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Formata data para exibição
 */
function format_date($date, $format = 'd/m/Y') {
    if (empty($date)) {
        return '';
    }
    
    try {
        $dt = new DateTime($date);
        return $dt->format($format);
    } catch (Exception $e) {
        return $date;
    }
}

/**
 * Formata data e hora para exibição
 */
function format_datetime($datetime, $format = 'd/m/Y H:i') {
    return format_date($datetime, $format);
}

/**
 * Retorna diferença de tempo em formato legível
 */
function time_ago($datetime) {
    if (empty($datetime)) {
        return '';
    }
    
    try {
        $now = new DateTime();
        $ago = new DateTime($datetime);
        $diff = $now->diff($ago);
        
        if ($diff->y > 0) {
            return $diff->y . ' ano' . ($diff->y > 1 ? 's' : '') . ' atrás';
        }
        if ($diff->m > 0) {
            return $diff->m . ' mês' . ($diff->m > 1 ? 'es' : '') . ' atrás';
        }
        if ($diff->d > 0) {
            return $diff->d . ' dia' . ($diff->d > 1 ? 's' : '') . ' atrás';
        }
        if ($diff->h > 0) {
            return $diff->h . ' hora' . ($diff->h > 1 ? 's' : '') . ' atrás';
        }
        if ($diff->i > 0) {
            return $diff->i . ' minuto' . ($diff->i > 1 ? 's' : '') . ' atrás';
        }
        
        return 'agora mesmo';
        
    } catch (Exception $e) {
        return $datetime;
    }
}

/**
 * Trunca texto com reticências
 */
function truncate($text, $length = 100, $suffix = '...') {
    if (empty($text)) {
        return '';
    }
    
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    
    return mb_substr($text, 0, $length) . $suffix;
}

/**
 * Converte array para query string
 */
function build_query_string($params) {
    return http_build_query($params);
}