<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';

/* === RBAC: exigir permissão para VER clientes === */
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('clientes','view');

// Pega o ID do cliente pela URL
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($id <= 0) {
    header("Location: " . BASE_URL . "/clientes");
    exit;
}

// Busca o cliente no banco
try {
    $cliente = fetch_one("SELECT * FROM clients WHERE id = ?", [$id], true);
    if (!$cliente) {
        header("Location: " . BASE_URL . "/clientes");
        exit;
    }

    /* === RBAC: se o escopo de VIEW for 'own', só permite ver se for dono
       (usa owner_user_id quando existir; senão, fallback para created_by) === */
    $scopeView = rbac_scope('clientes','view'); // 'none' | 'own' | 'all'
    if ($scopeView === 'own') {
        $me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
        $owner = isset($cliente['owner_user_id'])
            ? (int)$cliente['owner_user_id']
            : (int)($cliente['created_by'] ?? 0);
        if ($owner !== $me_id) {
            http_response_code(403);
            exit('Você não tem permissão para visualizar este cliente.');
        }
    }
} catch (Exception $e) {
    die("Erro ao carregar cliente: " . $e->getMessage());
}

$page_title = "Cliente: " . htmlspecialchars($cliente['nome_completo']);

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';

/* === RBAC: calcular se pode EDITAR este cliente, respeitando escopo
   (mesmo fallback de owner_user_id -> created_by) === */
$can_edit_this = false;
if (rbac_can('clientes','update')) {
    $scopeUpd = rbac_scope('clientes','update'); // 'none' | 'own' | 'all'
    if ($scopeUpd === 'all') {
        $can_edit_this = true;
    } elseif ($scopeUpd === 'own') {
        $me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
        $owner = isset($cliente['owner_user_id'])
            ? (int)$cliente['owner_user_id']
            : (int)($cliente['created_by'] ?? 0);
        $can_edit_this = ($owner === $me_id);
    }
}
?>

<style>
.client-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 2rem;
    border-radius: 15px;
    margin-bottom: 2rem;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}
.info-card {
    background: var(--bs-dark);
    border: 1px solid rgba(255,255,255,0.1);
    border-radius: 10px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}
.info-label {
    color: #adb5bd;
    font-size: 0.875rem;
    font-weight: 500;
    margin-bottom: 0.25rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}
.info-value {
    color: #e9ecef;
    font-size: 1.1rem;
    font-weight: 500;
    margin-bottom: 0;
}
.info-value a { color: #0dcaf0; text-decoration: none; }
.info-value a:hover { color: #0aa4c0; text-decoration: underline; }
.badge-custom { font-size: 0.875rem; padding: 0.5rem 1rem; }
.toolbar-fixed {
    position: fixed; bottom: 20px; right: 20px; z-index: 1050;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
}
@media (max-width: 768px) {
    .client-header { padding: 1.5rem; margin-bottom: 1rem; }
    .info-card { padding: 1rem; }
    .toolbar-fixed { bottom: 10px; right: 10px; left: 10px; }
    .toolbar-fixed .btn-group { width: 100%; flex-wrap: wrap; }
}
</style>

<div class="container mt-4 mb-5">
    <!-- Header do Cliente -->
    <div class="client-header">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1 class="mb-1"><?=htmlspecialchars($cliente['nome_completo'])?></h1>
                <div class="d-flex flex-wrap gap-2 mb-2">
                    <span class="badge bg-light text-dark">
                        <i class="bi bi-person me-1"></i>Cliente #<?=$cliente['id']?>
                    </span>
                    <?php
                    $tipos = [
                        'comprador' => ['bg-primary', 'Comprador'],
                        'vendedor' => ['bg-success', 'Vendedor'], 
                        'locatario' => ['bg-info', 'Locatário'],
                        'proprietario' => ['bg-warning', 'Proprietário']
                    ];
                    if (isset($tipos[$cliente['tipo_cliente']])) {
                        [$classe, $texto] = $tipos[$cliente['tipo_cliente']];
                        echo "<span class='badge $classe'>$texto</span>";
                    }
                    ?>
                    <?php if ($cliente['created_at']): ?>
                        <span class="badge bg-secondary">
                            <i class="bi bi-calendar me-1"></i>
                            <?=date('d/m/Y', strtotime($cliente['created_at']))?>
                        </span>
                    <?php endif; ?>
                </div>
                <?php if ($cliente['telefone_principal']): ?>
                    <div class="mt-2">
                        <i class="bi bi-telephone text-light me-2"></i>
                        <a href="tel:<?=htmlspecialchars($cliente['telefone_principal'])?>" 
                           class="text-light text-decoration-none">
                            <?=htmlspecialchars($cliente['telefone_principal'])?>
                        </a>
                        <?php if ($cliente['telefone_secundario']): ?>
                            <span class="mx-2">•</span>
                            <a href="tel:<?=htmlspecialchars($cliente['telefone_secundario'])?>" 
                               class="text-light text-decoration-none">
                                <?=htmlspecialchars($cliente['telefone_secundario'])?>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            <div class="col-md-4 text-md-end">
                <?php if ($cliente['email']): ?>
                    <a href="mailto:<?=htmlspecialchars($cliente['email'])?>" 
                       class="btn btn-outline-light btn-sm me-2">
                        <i class="bi bi-envelope me-1"></i>E-mail
                    </a>
                <?php endif; ?>
                <?php if ($cliente['telefone_principal']): ?>
                    <a href="https://wa.me/55<?=preg_replace('/\D/', '', $cliente['telefone_principal'])?>" 
                       target="_blank" class="btn btn-success btn-sm">
                        <i class="bi bi-whatsapp me-1"></i>WhatsApp
                    </a>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Informações Pessoais -->
        <div class="col-lg-6">
            <div class="info-card">
                <h5 class="mb-3 text-primary">
                    <i class="bi bi-person-circle me-2"></i>Informações Pessoais
                </h5>
                
                <?php if ($cliente['cpf_cnpj']): ?>
                <div class="mb-3">
                    <div class="info-label">CPF/CNPJ</div>
                    <div class="info-value"><?=htmlspecialchars($cliente['cpf_cnpj'])?></div>
                </div>
                <?php endif; ?>

                <?php if ($cliente['data_nascimento']): ?>
                <div class="mb-3">
                    <div class="info-label">Data de Nascimento</div>
                    <div class="info-value">
                        <?=date('d/m/Y', strtotime($cliente['data_nascimento']))?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($cliente['email']): ?>
                <div class="mb-3">
                    <div class="info-label">E-mail</div>
                    <div class="info-value">
                        <a href="mailto:<?=htmlspecialchars($cliente['email'])?>">
                            <?=htmlspecialchars($cliente['email'])?>
                        </a>
                    </div>
                </div>
                <?php endif; ?>

                <div class="mb-0">
                    <div class="info-label">Tipo de Cliente</div>
                    <div class="info-value">
                        <?php
                        $tipos = [
                            'comprador' => 'Comprador',
                            'vendedor' => 'Vendedor', 
                            'locatario' => 'Locatário',
                            'proprietario' => 'Proprietário'
                        ];
                        echo $tipos[$cliente['tipo_cliente']] ?? 'Cliente';
                        ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Contato -->
        <div class="col-lg-6">
            <div class="info-card">
                <h5 class="mb-3 text-success">
                    <i class="bi bi-telephone me-2"></i>Contato
                </h5>
                
                <div class="mb-3">
                    <div class="info-label">Telefone Principal</div>
                    <div class="info-value">
                        <i class="bi bi-whatsapp text-success me-2"></i>
                        <a href="tel:<?=htmlspecialchars($cliente['telefone_principal'])?>">
                            <?=htmlspecialchars($cliente['telefone_principal'])?>
                        </a>
                    </div>
                </div>

                <?php if ($cliente['telefone_secundario']): ?>
                <div class="mb-3">
                    <div class="info-label">Telefone Secundário</div>
                    <div class="info-value">
                        <a href="tel:<?=htmlspecialchars($cliente['telefone_secundario'])?>">
                            <?=htmlspecialchars($cliente['telefone_secundario'])?>
                        </a>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($cliente['created_at']): ?>
                <div class="mb-0">
                    <div class="info-label">Cadastrado em</div>
                    <div class="info-value">
                        <?=date('d/m/Y H:i', strtotime($cliente['created_at']))?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($cliente['updated_at'] && $cliente['updated_at'] != $cliente['created_at']): ?>
                <div class="mt-2 pt-2 border-top">
                    <div class="info-label">Última Atualização</div>
                    <div class="info-value text-muted">
                        <?=date('d/m/Y H:i', strtotime($cliente['updated_at']))?>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Endereço -->
        <?php if ($cliente['endereco'] || $cliente['cidade'] || $cliente['bairro']): ?>
        <div class="col-12">
            <div class="info-card">
                <h5 class="mb-3 text-info">
                    <i class="bi bi-geo-alt me-2"></i>Endereço
                </h5>
                
                <div class="row">
                    <?php if ($cliente['endereco']): ?>
                    <div class="col-md-8 mb-2">
                        <div class="info-label">Endereço</div>
                        <div class="info-value"><?=htmlspecialchars($cliente['endereco'])?></div>
                    </div>
                    <?php endif; ?>

                    <div class="col-md-4 mb-2">
                        <div class="info-label">CEP</div>
                        <div class="info-value"><?=htmlspecialchars($cliente['cep'])?></div>
                    </div>

                    <?php if ($cliente['bairro']): ?>
                    <div class="col-md-4 mb-2">
                        <div class="info-label">Bairro</div>
                        <div class="info-value"><?=htmlspecialchars($cliente['bairro'])?></div>
                    </div>
                    <?php endif; ?>

                    <div class="col-md-5 mb-2">
                        <div class="info-label">Cidade</div>
                        <div class="info-value"><?=htmlspecialchars($cliente['cidade'])?></div>
                    </div>

                    <div class="col-md-3 mb-2">
                        <div class="info-label">UF</div>
                        <div class="info-value"><?=htmlspecialchars($cliente['uf'])?></div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Observações -->
        <?php if ($cliente['observacoes']): ?>
        <div class="col-12">
            <div class="info-card">
                <h5 class="mb-3 text-warning">
                    <i class="bi bi-chat-text me-2"></i>Observações
                </h5>
                <div class="info-value" style="white-space: pre-wrap; line-height: 1.6;">
                    <?=nl2br(htmlspecialchars($cliente['observacoes']))?>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Barra de ferramentas flutuante -->
<div class="toolbar-fixed">
    <div class="btn-group shadow-lg" role="group">
        <?php if ($can_edit_this): ?>
        <a href="<?=BASE_URL?>/clientes/<?=$cliente['id']?>/editar" 
           class="btn btn-primary">
            <i class="bi bi-pencil me-2"></i>Editar
        </a>
        <?php endif; ?>
        <a href="<?=BASE_URL?>/clientes" 
           class="btn btn-secondary">
            <i class="bi bi-arrow-left me-2"></i>Voltar
        </a>
        <button class="btn btn-outline-light" onclick="window.print()">
            <i class="bi bi-printer me-2"></i>Imprimir
        </button>
    </div>
</div>

<script>
// Melhorar impressão
window.addEventListener('beforeprint', function() {
    var tb = document.querySelector('.toolbar-fixed');
    if (tb) tb.style.display = 'none';
});
window.addEventListener('afterprint', function() {
    var tb = document.querySelector('.toolbar-fixed');
    if (tb) tb.style.display = 'block';
});
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>