<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

/* Permissão para EXCLUIR imóveis */
rbac_require('imoveis','delete');

$BASE     = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$redirect = isset($_GET['redirect']) && $_GET['redirect'] ? $_GET['redirect'] : "{$BASE}/imoveis";
$id       = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
if ($id <= 0) { header("Location: {$redirect}"); exit; }

/* Carrega imóvel sem citar colunas opcionais */
try {
  $imovel = fetch_one("SELECT id, titulo FROM properties WHERE id = ?", [$id], true);
} catch (Throwable $e) {
  http_response_code(500); exit('Erro ao carregar imóvel.');
}
if (!$imovel) { header("Location: {$redirect}"); exit; }

/* Descobrir colunas opcionais para checar escopo */
$has_created_by = false;
try { $c = fetch_one("SHOW COLUMNS FROM properties LIKE 'created_by'", [], true); $has_created_by = (bool)$c; } catch (Throwable $e) {}

$scope = rbac_scope('imoveis','delete'); // 'none' | 'own' | 'all'
if ($scope === 'own') {
  // Se existir created_by, valida; se não existir, bloqueia por segurança
  if ($has_created_by) {
    $me = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
    $row = fetch_one("SELECT created_by FROM properties WHERE id = ?", [$id], true);
    if (!isset($row['created_by']) || (int)$row['created_by'] !== $me) {
      http_response_code(403); exit('Você não tem permissão para excluir este imóvel.');
    }
  } else {
    http_response_code(403); exit('Permissão insuficiente para excluir (campo responsável ausente).');
  }
}

/* CSRF simples */
function csrf_token_get() {
  if (function_exists('csrf_field')) return null;
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check_local($t) {
  if (function_exists('csrf_check')) return csrf_check();
  return isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], (string)$t);
}

/* Remoção recursiva de pasta */
function rrmdir_silent($dir) {
  if (!is_dir($dir)) return;
  $items = @scandir($dir); if ($items === false) return;
  foreach ($items as $it) {
    if ($it === '.' || $it === '..') continue;
    $p = $dir . DIRECTORY_SEPARATOR . $it;
    if (is_dir($p)) rrmdir_silent($p); else @unlink($p);
  }
  @rmdir($dir);
}

/* POST: confirmar exclusão */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $ok = csrf_check_local($_POST['csrf'] ?? '');
  if (!$ok) { http_response_code(400); exit('CSRF inválido.'); }

  $upload_dir_fs = __DIR__ . '/../../uploads/imoveis/' . (int)$imovel['id'] . '/';

  try {
    begin_transaction();

    // 1) Excluir arquivos do disco
    rrmdir_silent($upload_dir_fs);

    // 2) Excluir dependências (ignora se tabela não existir)
    try { execute("DELETE FROM property_photos WHERE property_id = ?", [(int)$imovel['id']]); } catch (Throwable $e) {}
    try { execute("DELETE FROM property_feature_values WHERE property_id = ?", [(int)$imovel['id']]); } catch (Throwable $e) {}
    try { execute("DELETE FROM property_logs WHERE property_id = ?", [(int)$imovel['id']]); } catch (Throwable $e) {}

    // 3) Registrar log de exclusão (se existir a tabela)
    try {
      $user_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
      execute(
        "INSERT INTO property_logs (property_id, user_id, acao, detalhes) VALUES (?, ?, 'excluido', ?)",
        [(int)$imovel['id'], $user_id, 'Exclusão do imóvel "' . ($imovel['titulo'] ?? '') . '"']
      );
    } catch (Throwable $e) {}

    // 4) Excluir o imóvel
    execute("DELETE FROM properties WHERE id = ?", [(int)$imovel['id']]);

    commit();
    header("Location: {$redirect}?deleted=1");
    exit;

  } catch (Throwable $e) {
    rollback();
    http_response_code(500); exit('Erro ao excluir imóvel.');
  }
}

/* GET: confirmação */
$page_title = 'Excluir imóvel';
include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
$csrf = csrf_token_get();
?>
<div class="container my-4">
  <div class="card border-danger">
    <div class="card-header bg-danger text-white">
      <i class="bi bi-trash"></i> Confirmar exclusão
    </div>
    <div class="card-body">
      <p>Tem certeza de que deseja excluir o imóvel <strong><?= htmlspecialchars($imovel['titulo'] ?? ("#".$id)) ?></strong> (ID <?= (int)$imovel['id'] ?>)?</p>
      <div class="alert alert-warning">
        <i class="bi bi-exclamation-triangle"></i> Esta ação não pode ser desfeita. Fotos e vínculos serão removidos.
      </div>
      <form method="post" class="d-flex gap-2">
        <input type="hidden" name="id" value="<?= (int)$imovel['id'] ?>">
        <?php if ($csrf): ?>
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
        <?php elseif (function_exists('csrf_field')): ?>
          <?= csrf_field(); ?>
        <?php endif; ?>
        <a href="<?= htmlspecialchars($redirect) ?>" class="btn btn-secondary">
          <i class="bi bi-arrow-left"></i> Cancelar
        </a>
        <button class="btn btn-danger">
          <i class="bi bi-trash"></i> Excluir definitivamente
        </button>
      </form>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>