<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Novo Atendimento';
$active     = 'atendimentos';

rbac_require('atendimentos','create');

$BASE  = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* Status válidos */
$STATUS = ['novo','em_atendimento','negociacao','finalizado'];

/* Helpers locais */
if (!function_exists('format_money')) {
  function format_money($v) { return 'R$ ' . number_format((float)$v, 2, ',', '.'); }
}

/* Quem pode escolher o responsável?
   Regra: admin OU escopo 'create' = 'all' em atendimentos OU visão 'all' em clientes. */
$canAssignOwner = (function_exists('rbac_is_admin') && rbac_is_admin())
               || (rbac_scope('atendimentos','create') === 'all')
               || (rbac_scope('clientes','view') === 'all');

$erro = null;

/* Detecta se existe a coluna property_id em interactions (para exibir o campo) */
$hasPropertyCol = false;
try { fetch_one("SELECT property_id FROM interactions LIMIT 1", [], true); $hasPropertyCol = true; } catch (Throwable $e) {}

/* POST: salvar */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) {
      throw new Exception('CSRF inválido.');
    }

    $client_id   = (int)($_POST['client_id'] ?? 0);
    $status      = (string)($_POST['status'] ?? 'novo');
    $nota        = trim($_POST['nota'] ?? '');
    $next_at     = trim($_POST['next_followup_at'] ?? '');
    $property_id = (int)($_POST['property_id'] ?? 0); // opcional
    $owner_id    = $canAssignOwner ? (int)($_POST['owner_user_id'] ?? 0) : $me_id;

    if ($client_id <= 0) { throw new Exception('Selecione um cliente.'); }
    if (!in_array($status, $STATUS, true)) { throw new Exception('Status inválido.'); }
    if ($canAssignOwner && $owner_id <= 0) { throw new Exception('Selecione o responsável.'); }

    // valida se o responsável existe
    $ownerRow = fetch_one("SELECT id,nome FROM users WHERE id = ?", [$owner_id], true);
    if (!$ownerRow) { throw new Exception('Responsável inválido.'); }

    // Checagem de acesso ao cliente:
    // - Se admin/gerente com visão ALL (ou canAssignOwner), pode qualquer cliente.
    // - Caso contrário, precisa ser "meu" (owner_user_id=eu OU created_by=eu quando não tem owner).
    $hasAllClients = (rbac_scope('clientes','view') === 'all') || $canAssignOwner;
    if ($hasAllClients) {
      $check = fetch_one("SELECT id FROM clients WHERE id = ?", [$client_id], true);
    } else {
      $check = fetch_one(
        "SELECT id FROM clients 
          WHERE id = ? 
            AND (owner_user_id = ? OR (owner_user_id IS NULL AND created_by = ?))",
        [$client_id, $me_id, $me_id],
        true
      );
    }
    if (!$check) { throw new Exception('Você não pode registrar atendimento para este cliente.'); }

    // valida imóvel se informado
    if ($hasPropertyCol && $property_id > 0) {
      $p = fetch_one("SELECT id FROM properties WHERE id = ?", [$property_id], true);
      if (!$p) { throw new Exception('Imóvel selecionado inválido.'); }
    }

    // normaliza próxima ação
    $next_dt = ($next_at !== '') ? date('Y-m-d H:i:s', strtotime($next_at)) : null;

    // INSERT com/sem property_id dependendo do schema
    if ($hasPropertyCol) {
      execute(
        "INSERT INTO interactions (client_id, property_id, status, nota, owner_user_id, created_by, next_followup_at, created_at, updated_at)
         VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())",
        [
          $client_id,
          ($property_id > 0 ? $property_id : null),
          $status,
          ($nota !== '' ? $nota : null),
          $owner_id ?: null,
          $me_id ?: null,
          $next_dt
        ]
      );
    } else {
      execute(
        "INSERT INTO interactions (client_id, status, nota, owner_user_id, created_by, next_followup_at, created_at, updated_at)
         VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())",
        [
          $client_id,
          $status,
          ($nota !== '' ? $nota : null),
          $owner_id ?: null,
          $me_id ?: null,
          $next_dt
        ]
      );
    }

    header("Location: {$BASE}/atendimentos?ok=1");
    exit;

  } catch (Throwable $e) {
    $erro = $e->getMessage();
  }
}

/* Carregar clientes (ALL se tiver visão all, senão os “meus”) */
try {
  $scope_cli = rbac_scope('clientes','view'); // 'own' | 'all'
  $params = [];
  $sqlCli = "SELECT id, nome_completo FROM clients";
  if ($scope_cli === 'own') {
    $sqlCli .= " WHERE (owner_user_id = ? OR (owner_user_id IS NULL AND created_by = ?))";
    $params = [$me_id, $me_id];
  }
  $sqlCli .= " ORDER BY nome_completo ASC LIMIT 500";
  $clientes = fetch_all($sqlCli, $params, true);
} catch (Throwable $e) {
  $erro = $erro ?: $e->getMessage();
  $clientes = [];
}

/* Carregar responsáveis (lista de usuários) — só precisa se $canAssignOwner */
$owners = [];
if ($canAssignOwner) {
  try {
    $owners = fetch_all("SELECT id, nome FROM users ORDER BY nome ASC", [], true) ?: [];
  } catch (Throwable $e) {
    $owners = [];
  }
}

/* Carregar imóveis (campo opcional) */
$imoveis = [];
if ($hasPropertyCol) {
  try {
    $imoveis = fetch_all("
      SELECT id, titulo, cidade, uf, tipo, finalidade, preco
        FROM properties
    ORDER BY updated_at DESC, id DESC
       LIMIT 300
    ", [], true) ?: [];
  } catch (Throwable $e) { $imoveis = []; }
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0"><i class="bi bi-clipboard2-plus text-primary"></i> Novo Atendimento</h1>
    <a href="<?=$BASE?>/atendimentos" class="btn btn-outline-secondary"><i class="bi bi-arrow-left"></i> Voltar</a>
  </div>

  <?php if ($erro): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($erro) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <form class="card card-body" method="post" novalidate>
    <input type="hidden" name="csrf" value="<?= htmlspecialchars($CSRF) ?>">

    <div class="row g-3">
      <div class="col-md-6">
        <label class="form-label">Cliente *</label>
        <select name="client_id" class="form-select" required>
          <option value="">Selecione...</option>
          <?php foreach ($clientes as $c): ?>
            <option value="<?=$c['id']?>" <?= (isset($_POST['client_id']) && (int)$_POST['client_id']===$c['id'])?'selected':'' ?>>
              <?= htmlspecialchars($c['nome_completo']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">Apenas seus clientes (ou todos, se você for gerente/admin).</div>
      </div>

      <?php if ($canAssignOwner): ?>
      <div class="col-md-6">
        <label class="form-label">Responsável *</label>
        <select name="owner_user_id" class="form-select" required>
          <option value="">Selecione o responsável...</option>
          <?php
            $post_owner = isset($_POST['owner_user_id']) ? (int)$_POST['owner_user_id'] : 0;
            foreach ($owners as $o):
          ?>
            <option value="<?=$o['id']?>" <?= ($post_owner===$o['id'])?'selected':'' ?>>
              <?= htmlspecialchars($o['nome']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">Quem acompanhará este atendimento (corretor).</div>
      </div>
      <?php else: ?>
        <!-- mantém responsável fixo no back-end (owner = $me_id) -->
      <?php endif; ?>

      <?php if ($hasPropertyCol): ?>
      <div class="col-md-12">
        <label class="form-label">Imóvel (opcional)</label>
        <select name="property_id" class="form-select">
          <option value="">— (nenhum) —</option>
          <?php
            $post_property_id = isset($_POST['property_id']) ? (int)$_POST['property_id'] : 0;
            foreach ($imoveis as $p):
              $label = sprintf(
                "#%d · %s — %s/%s · %s/%s · %s",
                (int)$p['id'],
                (string)$p['titulo'],
                (string)$p['cidade'],
                strtoupper((string)$p['uf']),
                strtoupper((string)$p['tipo']),
                strtoupper((string)$p['finalidade']),
                format_money((float)$p['preco'])
              );
          ?>
            <option value="<?=$p['id']?>" <?= ($post_property_id===$p['id'])?'selected':'' ?>>
              <?= htmlspecialchars($label) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">Use quando o cliente demonstrar interesse direto por um imóvel.</div>
      </div>
      <?php endif; ?>

      <div class="col-md-3">
        <label class="form-label">Status *</label>
        <select name="status" class="form-select" required>
          <?php
          $post_status = $_POST['status'] ?? 'novo';
          $labels = [
            'novo'            => 'Novo',
            'em_atendimento'  => 'Em Atendimento',
            'negociacao'      => 'Negociação',
            'finalizado'      => 'Finalizado'
          ];
          foreach ($STATUS as $st):
          ?>
            <option value="<?=$st?>" <?=($post_status===$st)?'selected':''?>><?=$labels[$st]?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label">Próxima ação (data/hora)</label>
        <input type="datetime-local" name="next_followup_at" class="form-control"
               value="<?= htmlspecialchars($_POST['next_followup_at'] ?? '') ?>">
        <div class="form-text">Opcional — usado para lembretes/atrasos.</div>
      </div>

      <div class="col-12">
        <label class="form-label">Nota</label>
        <textarea name="nota" class="form-control" rows="4"
          placeholder="Ex.: liguei para o cliente, pediu retorno amanhã..."><?= htmlspecialchars($_POST['nota'] ?? '') ?></textarea>
      </div>
    </div>

    <div class="d-flex justify-content-end gap-2 mt-4">
      <a class="btn btn-outline-secondary" href="<?=$BASE?>/atendimentos">
        <i class="bi bi-x-circle"></i> Cancelar
      </a>
      <button class="btn btn-primary">
        <i class="bi bi-check-circle"></i> Salvar
      </button>
    </div>
  </form>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>