<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';
$page_title = 'Fotos do Imóvel';

// Verificar se ID foi passado
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: " . BASE_URL . "/imoveis");
    exit;
}

$property_id = intval($_GET['id']);

// Verificar se o imóvel existe
$imovel = fetch_one("SELECT titulo FROM properties WHERE id = ?", [$property_id], true);
if (!$imovel) {
    header("Location: " . BASE_URL . "/imoveis");
    exit;
}

$imovel_titulo = $imovel['titulo'];

// Processar upload de fotos
$mensagem = null;
$tipo_mensagem = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_FILES['fotos']) && !empty($_FILES['fotos']['name'][0])) {
        try {
            $upload_dir = __DIR__ . '/../../uploads/imoveis/' . $property_id . '/';
            
            // Criar pasta se não existir
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $fotos_uploaded = 0;
            $erros_upload = [];
            
            foreach ($_FILES['fotos']['name'] as $key => $name) {
                if ($_FILES['fotos']['error'][$key] == 0) {
                    $file_tmp = $_FILES['fotos']['tmp_name'][$key];
                    $file_size = $_FILES['fotos']['size'][$key];
                    $file_ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
                    
                    // Validar extensão
                    $allowed_ext = ['jpg', 'jpeg', 'png', 'webp'];
                    if (!in_array($file_ext, $allowed_ext)) {
                        $erros_upload[] = "Arquivo $name: Extensão não permitida";
                        continue;
                    }
                    
                    // Validar tamanho (máx 5MB)
                    if ($file_size > 5 * 1024 * 1024) {
                        $erros_upload[] = "Arquivo $name: Tamanho muito grande (máx 5MB)";
                        continue;
                    }
                    
                    // Gerar nome único
                    $filename = uniqid() . '.' . $file_ext;
                    $filepath = $upload_dir . $filename;
                    $webpath = BASE_URL . '/uploads/imoveis/' . $property_id . '/' . $filename;
                    
                    // Mover arquivo
                    if (move_uploaded_file($file_tmp, $filepath)) {
                        // Salvar no banco
                        $sql = "INSERT INTO property_photos (property_id, filename, path, is_main, ordem) 
                                VALUES (?, ?, ?, 0, 0)";
                        execute($sql, [$property_id, $filename, $webpath]);
                        $fotos_uploaded++;
                    } else {
                        $erros_upload[] = "Erro ao mover arquivo $name";
                    }
                }
            }
            
            if ($fotos_uploaded > 0) {
                $mensagem = "$fotos_uploaded foto(s) enviada(s) com sucesso!";
                $tipo_mensagem = 'success';
            }
            
            if (!empty($erros_upload)) {
                $mensagem .= "<br><small class='text-danger'>" . implode('<br>', $erros_upload) . "</small>";
            }
            
        } catch (Exception $e) {
            $mensagem = "Erro ao processar upload: " . $e->getMessage();
            $tipo_mensagem = 'danger';
        }
    }
    
    // Processar definir foto principal
    if (isset($_POST['definir_principal']) && is_numeric($_POST['definir_principal'])) {
        try {
            // Resetar todas as fotos como não principal
            execute("UPDATE property_photos SET is_main = 0 WHERE property_id = ?", [$property_id]);
            
            // Definir a selecionada como principal
            $foto_id = intval($_POST['definir_principal']);
            execute("UPDATE property_photos SET is_main = 1 WHERE id = ? AND property_id = ?", 
                   [$foto_id, $property_id]);
            
            $mensagem = "Foto principal definida com sucesso!";
            $tipo_mensagem = 'success';
            
        } catch (Exception $e) {
            $mensagem = "Erro ao definir foto principal: " . $e->getMessage();
            $tipo_mensagem = 'danger';
        }
    }
    
    // Processar exclusão de foto
    if (isset($_POST['excluir_foto']) && is_numeric($_POST['excluir_foto'])) {
        try {
            $foto_id = intval($_POST['excluir_foto']);
            $foto = fetch_one("SELECT path, filename FROM property_photos WHERE id = ? AND property_id = ?", 
                             [$foto_id, $property_id], true);
            
            if ($foto) {
                // Deletar arquivo físico
                $file_path = __DIR__ . '/../../' . str_replace(BASE_URL, '', $foto['path']);
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
                
                // Deletar do banco
                execute("DELETE FROM property_photos WHERE id = ? AND property_id = ?", 
                       [$foto_id, $property_id]);
                
                $mensagem = "Foto excluída com sucesso!";
                $tipo_mensagem = 'success';
            }
            
        } catch (Exception $e) {
            $mensagem = "Erro ao excluir foto: " . $e->getMessage();
            $tipo_mensagem = 'danger';
        }
    }
}

// Buscar fotos do imóvel
$fotos = fetch_all("SELECT * FROM property_photos WHERE property_id = ? ORDER BY ordem ASC, created_at ASC", 
                  [$property_id], true);

// Contar total de fotos
$total_fotos = count($fotos);
$foto_principal = array_filter($fotos, function($foto) { return $foto['is_main'] == 1; });
$foto_principal = !empty($foto_principal) ? reset($foto_principal) : null;

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<div class="container mt-4 mb-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="fw-bold mb-1">
                <i class="bi bi-images text-primary me-2"></i>
                Fotos - <?=htmlspecialchars($imovel_titulo)?>
            </h1>
            <p class="text-muted mb-0">Gerencie as fotos do imóvel (<?= $total_fotos ?> foto<?= $total_fotos != 1 ? 's' : '' ?>)</p>
        </div>
        <div class="d-flex gap-2">
            <a href="<?=BASE_URL?>/imoveis" class="btn btn-outline-secondary">
                <i class="bi bi-list me-2"></i>Ver Imóveis
            </a>
            <a href="<?=BASE_URL?>/imoveis/<?= $property_id ?>/editar" class="btn btn-outline-primary">
                <i class="bi bi-pencil me-2"></i>Editar Imóvel
            </a>
        </div>
    </div>

    <?php if ($mensagem): ?>
        <div class="alert alert-<?= $tipo_mensagem ?> alert-dismissible fade show" role="alert">
            <?= $mensagem ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Upload de Fotos -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0"><i class="bi bi-cloud-upload me-2"></i>Adicionar Novas Fotos</h5>
        </div>
        <div class="card-body">
            <form method="POST" enctype="multipart/form-data">
                <div class="mb-3">
                    <label class="form-label">Selecionar Fotos (Máx 5MB cada, JPG/PNG/WEBP)</label>
                    <input type="file" name="fotos[]" class="form-control" multiple accept="image/*" required>
                    <div class="form-text">Você pode selecionar múltiplas fotos de uma vez. Máximo 20 fotos por imóvel.</div>
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-upload me-2"></i>Enviar Fotos
                </button>
            </form>
        </div>
    </div>

    <!-- Status da Foto Principal -->
    <div class="alert alert-info mb-4">
        <div class="d-flex align-items-center">
            <i class="bi bi-star-fill text-warning me-2 fs-5"></i>
            <div>
                <strong>Foto Principal:</strong> 
                <?php if ($foto_principal): ?>
                    Definida (<?= $total_fotos > 1 ? $total_fotos - 1 : '0' ?> foto<?= $total_fotos > 2 ? 's' : '' ?> secundária<?= $total_fotos > 2 ? 's' : '' ?>)
                <?php else: ?>
                    <span class="text-warning">Nenhuma foto principal definida</span>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Galeria de Fotos -->
    <?php if (!empty($fotos)): ?>
        <div class="row g-3">
            <?php foreach ($fotos as $index => $foto): ?>
                <div class="col-md-3 col-sm-6 mb-3">
                    <div class="card h-100 position-relative">
                        <img src="<?= htmlspecialchars($foto['path']) ?>" 
                             class="card-img-top" 
                             style="height: 200px; object-fit: cover;"
                             alt="Foto <?= $index + 1 ?>"
                             onerror="this.src='<?= BASE_URL ?>/assets/img/no-image.jpg'">
                        
                        <?php if ($foto['is_main']): ?>
                            <span class="position-absolute top-0 start-0 badge bg-warning rounded-pill m-2">
                                <i class="bi bi-star-fill me-1"></i>Principal
                            </span>
                        <?php endif; ?>
                        
                        <div class="card-body d-flex flex-column">
                            <h6 class="card-title mb-2">Foto <?= $index + 1 ?></h6>
                            <small class="text-muted mb-2 flex-grow-1">
                                <?= format_date($foto['created_at'], 'd/m/Y H:i') ?>
                            </small>
                            
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="definir_principal" value="<?= $foto['id'] ?>">
                                <button type="submit" class="btn btn-sm btn-outline-warning w-100 mb-1" 
                                        <?= $foto['is_main'] ? 'disabled' : '' ?>>
                                    <i class="bi bi-star<?= $foto['is_main'] ? '-fill' : '' ?> me-1"></i>
                                    <?= $foto['is_main'] ? 'Principal' : 'Definir Principal' ?>
                                </button>
                            </form>
                            
                            <form method="POST" class="d-inline" onsubmit="return confirm('Excluir esta foto?')">
                                <input type="hidden" name="excluir_foto" value="<?= $foto['id'] ?>">
                                <button type="submit" class="btn btn-sm btn-outline-danger w-100">
                                    <i class="bi bi-trash me-1"></i>Excluir
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php else: ?>
        <div class="text-center py-5">
            <i class="bi bi-images display-1 text-muted mb-3"></i>
            <h4 class="text-muted">Nenhuma foto cadastrada</h4>
            <p class="text-muted">Use a seção acima para adicionar fotos ao imóvel.</p>
        </div>
    <?php endif; ?>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>
