<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

/* ===== RBAC ===== */
rbac_require('admin','transfer_owner');

$BASE  = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* ===== CSRF ===== */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* ===== Helpers ===== */
if (!function_exists('format_date')) {
  function format_date($dt,$fmt='d/m/Y H:i'){ return $dt ? date($fmt, strtotime($dt)) : '-'; }
}

/* ===== Filtros ===== */
$q         = trim($_GET['q'] ?? '');
$owner_id  = ($_GET['owner_id'] ?? '') !== '' ? (int)$_GET['owner_id'] : null;

/* ===== Ações (POST): Transferir responsável ===== */
$ok_msg = null; $err_msg = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'transfer') {
  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) {
      throw new Exception('CSRF inválido.');
    }
    // Permissão já garantida por rbac_require acima

    $new_owner_id = (int)($_POST['new_owner_id'] ?? 0);
    $ids          = array_filter(array_map('intval', (array)($_POST['ids'] ?? [])));

    if ($new_owner_id <= 0) { throw new Exception('Selecione o novo responsável.'); }
    if (!$ids)               { throw new Exception('Selecione ao menos um cliente para reatribuir.'); }

    // Verifica se usuário de destino existe
    $newOwner = fetch_one("SELECT id, nome FROM users WHERE id = ?", [$new_owner_id], true);
    if (!$newOwner) { throw new Exception('Responsável (destino) inválido.'); }

    // Busca clientes selecionados com owner atual
    $in = implode(',', array_fill(0, count($ids), '?'));
    $sel = fetch_all("
      SELECT id, owner_user_id
        FROM clients
       WHERE id IN ($in)
    ", $ids, true);

    if (!$sel) { throw new Exception('Nenhum cliente válido encontrado.'); }

    // Detecta se a coluna changed_by existe em ownership_transfers
    $hasChangedBy = true;
    try {
      fetch_one("SELECT changed_by FROM ownership_transfers LIMIT 1", [], true);
    } catch (Throwable $e) {
      $hasChangedBy = false;
    }

    // Transação
    $pdo = db();
    $pdo->beginTransaction();

    // Atualiza donos
    $upd = $pdo->prepare("UPDATE clients SET owner_user_id = ?, updated_at = NOW() WHERE id = ?");

    // Prepara log com/sem changed_by
    if ($hasChangedBy) {
      $log = $pdo->prepare("
        INSERT INTO ownership_transfers (client_id, from_user_id, to_user_id, changed_by, changed_at)
        VALUES (?, ?, ?, ?, NOW())
      ");
    } else {
      $log = $pdo->prepare("
        INSERT INTO ownership_transfers (client_id, from_user_id, to_user_id, changed_at)
        VALUES (?, ?, ?, NOW())
      ");
    }

    $changed = 0;
    foreach ($sel as $c) {
      $from = (int)($c['owner_user_id'] ?? 0);
      if ($from === $new_owner_id) {
        // já é do mesmo; pula
        continue;
      }
      $upd->execute([$new_owner_id, (int)$c['id']]);
      if ($hasChangedBy) {
        $log->execute([(int)$c['id'], $from ?: null, $new_owner_id, $me_id ?: null]);
      } else {
        $log->execute([(int)$c['id'], $from ?: null, $new_owner_id]);
      }
      $changed++;
    }

    $pdo->commit();
    $ok_msg = $changed > 0
      ? "Reatribuição concluída: {$changed} cliente(s) atualizados para “{$newOwner['nome']}”."
      : "Nenhum cliente precisou ser alterado (já estavam com o responsável escolhido).";

  } catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) { $pdo->rollBack(); }
    $err_msg = $e->getMessage();
  }
}

/* ===== Donos (para filtro e destino) ===== */
$owners = fetch_all("
  SELECT id, nome
    FROM users
ORDER BY nome ASC
", [], true) ?: [];

/* ===== WHERE da listagem ===== */
$where  = []; $params = [];
if ($q !== '') {
  $where[]  = "c.nome_completo LIKE ?";
  $params[] = "%{$q}%";
}
if (!is_null($owner_id)) {
  $where[]  = "c.owner_user_id = ?";
  $params[] = $owner_id;
}
$sqlWhere = $where ? ('WHERE '.implode(' AND ', $where)) : '';

/* ===== Lista de clientes (limite razoável) ===== */
$rows = fetch_all("
  SELECT c.id, c.nome_completo, c.owner_user_id, c.updated_at, u.nome AS owner_nome
    FROM clients c
LEFT JOIN users u ON u.id = c.owner_user_id
  $sqlWhere
ORDER BY c.updated_at DESC, c.id DESC
  LIMIT 300
", $params, true) ?: [];

$page_title = 'Admin · Reatribuir responsável';
include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
.admin-toolbar .btn,
.filter-bar .form-control,
.filter-bar .form-select { height: 38px; }
.table thead th { white-space: nowrap; }
.text-small-muted { color:#adb5bd; font-size:.85rem; }
.sticky-actions { position: sticky; bottom: 0; background: var(--bs-dark); padding: .75rem; border-top:1px solid rgba(255,255,255,.1); }
</style>

<div class="container my-4">
  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center mb-3">
    <div class="d-flex align-items-center gap-2">
      <i class="bi bi-shuffle" style="font-size:1.4rem; color: var(--primary);"></i>
      <div>
        <h1 class="h4 m-0">Reatribuir responsável</h1>
        <div class="text-muted small">Busque o cliente e transfira o responsável sem sair desta tela.</div>
      </div>
    </div>
    <div class="admin-toolbar d-flex gap-2">
      <a href="<?=$BASE?>/admin" class="btn btn-outline-secondary"><i class="bi bi-speedometer2 me-1"></i> Admin</a>
    </div>
  </div>

  <?php if ($ok_msg): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i><?=$ok_msg?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>
  <?php if ($err_msg): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i><?=htmlspecialchars($err_msg)?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <!-- FILTROS -->
  <form class="card card-body mb-3 filter-bar" method="get">
    <div class="row row-cols-1 row-cols-md-auto g-2 align-items-end">
      <div class="col">
        <label class="form-label">Cliente</label>
        <input type="text" name="q" class="form-control" placeholder="Nome do cliente" value="<?=htmlspecialchars($q)?>">
      </div>
      <div class="col">
        <label class="form-label">Responsável</label>
        <select name="owner_id" class="form-select">
          <option value="">— Todos —</option>
          <?php foreach ($owners as $o): ?>
            <option value="<?=$o['id']?>" <?=$owner_id===$o['id']?'selected':''?>><?=htmlspecialchars($o['nome'])?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col">
        <label class="form-label d-none d-md-block">&nbsp;</label>
        <button class="btn btn-outline-primary"><i class="bi bi-search"></i></button>
        <a class="btn btn-outline-secondary" href="<?=$BASE?>/admin/reatribuir"><i class="bi bi-x-circle"></i></a>
      </div>
    </div>
  </form>

  <!-- LISTA + TRANSFERÊNCIA EM MASSA -->
  <form method="post" class="card">
    <input type="hidden" name="csrf" value="<?=$CSRF?>">
    <input type="hidden" name="action" value="transfer">

    <div class="table-responsive">
      <table class="table table-dark table-hover align-middle mb-0">
        <thead>
          <tr>
            <th style="width:40px">
              <input type="checkbox" id="checkAll">
            </th>
            <th>Cliente</th>
            <th style="width:260px">Responsável atual</th>
            <th style="width:180px">Atualizado</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td>
                <input type="checkbox" class="row-check" name="ids[]" value="<?=$r['id']?>">
              </td>
              <td>
                <strong><?=htmlspecialchars($r['nome_completo'])?></strong>
                <div class="text-small-muted">#<?=$r['id']?></div>
              </td>
              <td><?=htmlspecialchars($r['owner_nome'] ?? '—')?></td>
              <td><?= format_date($r['updated_at']) ?></td>
            </tr>
          <?php endforeach; if (!$rows): ?>
            <tr><td colspan="4" class="text-center text-muted py-4">Nenhum cliente encontrado</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="sticky-actions d-flex flex-column flex-md-row gap-2 align-items-md-center justify-content-between">
      <div class="d-flex align-items-center gap-2">
        <label class="form-label m-0">Transferir selecionados para:</label>
        <select name="new_owner_id" class="form-select" style="min-width:260px">
          <option value="">— Selecione o novo responsável —</option>
          <?php foreach ($owners as $o): ?>
            <option value="<?=$o['id']?>"><?=htmlspecialchars($o['nome'])?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="d-flex gap-2">
        <button class="btn btn-success">
          <i class="bi bi-arrow-left-right me-1"></i>Transferir responsável
        </button>
        <button type="button" id="btnNone" class="btn btn-outline-light">Limpar seleção</button>
      </div>
    </div>
  </form>
</div>

<script>
(function(){
  const checkAll = document.getElementById('checkAll');
  const rows     = Array.from(document.querySelectorAll('.row-check'));
  const btnNone  = document.getElementById('btnNone');

  if (checkAll) {
    checkAll.addEventListener('change', ()=>{
      rows.forEach(chk => chk.checked = checkAll.checked);
    });
  }
  if (btnNone) {
    btnNone.addEventListener('click', ()=> {
      rows.forEach(chk => chk.checked = false);
      if (checkAll) checkAll.checked = false;
    });
  }
})();
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>