<?php
// modules/imoveis/ordenar.php — ordenar fotos por cliques (leve e rápido)
error_reporting(E_ALL);
ini_set('display_errors', 1);

if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('imoveis','update');

$BASE        = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title  = 'Ordenar Fotos';
$property_id = (int)($_GET['id'] ?? 0);
if ($property_id <= 0) { header("Location: {$BASE}/imoveis"); exit; }

/* === POST: salvar ordem === */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['order'])) {
    $csv = trim((string)$_POST['order']);
    $completeRest = isset($_POST['complete_rest']) && $_POST['complete_rest'] === '1';

    // Busca todos os IDs atuais (na ordem atual)
    $allRows = fetch_all("SELECT id FROM property_photos WHERE property_id = ? ORDER BY COALESCE(ordem,0), id", [$property_id], true) ?: [];
    $allIds  = array_map(static fn($r)=> (int)$r['id'], $allRows);

    // Sanitiza sequência clicada
    $picked = array_values(array_unique(array_filter(array_map('intval', explode(',', $csv)))));
    $picked = array_values(array_intersect($picked, $allIds)); // só ids que existem

    // Monta ordem final
    $final = $picked;
    if ($completeRest) {
        $rest = array_values(array_diff($allIds, $picked));
        $final = array_merge($final, $rest);
    }
    if (empty($final)) {
        header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=Sem+alteracoes");
        exit;
    }

    // Gerar ordens em "saltos de 10" (facilita futuras inserções)
    $ordemMap = [];
    $o = 10;
    foreach ($final as $fid) { $ordemMap[(int)$fid] = $o; $o += 10; }

    // Atualiza em lote via CASE WHEN
    $caseSql = []; $params = [];
    foreach ($ordemMap as $fid => $ord) { $caseSql[] = "WHEN ? THEN ?"; $params[] = $fid; $params[] = $ord; }
    $idsPlace = implode(',', array_fill(0, count($ordemMap), '?'));
    $sql = "UPDATE property_photos
               SET ordem = CASE id ".implode(' ', $caseSql)." ELSE ordem END
             WHERE property_id = ? AND id IN ($idsPlace)";
    $params[] = $property_id;
    foreach (array_keys($ordemMap) as $fid) { $params[] = $fid; }

    try {
        execute($sql, $params);
        header("Location: {$BASE}/imoveis/{$property_id}/fotos?ok=ordem");
        exit;
    } catch (Throwable $e) {
        // Se sua tabela ainda não tiver coluna 'ordem'
        header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=Coluna+ordem+ausente+ou+erro+ao+salvar");
        exit;
    }
}

/* === Carrega fotos (somente o necessário) === */
$rows = []; $hasPath = true;
try {
    $rows = fetch_all(
        "SELECT id, filename, path, COALESCE(ordem,0) AS ordem
           FROM property_photos
          WHERE property_id = ?
       ORDER BY COALESCE(ordem,0), id",
        [$property_id], true
    ) ?: [];
} catch (Throwable $e) {
    $hasPath = false;
    $rows = fetch_all(
        "SELECT id, filename, COALESCE(ordem,0) AS ordem
           FROM property_photos
          WHERE property_id = ?
       ORDER BY id",
        [$property_id], true
    ) ?: [];
}

$upload_dir_web = rtrim($BASE, '/') . '/uploads/imoveis/' . $property_id . '/';
$thumb = function(array $r) use ($upload_dir_web, $hasPath) {
    if ($hasPath && !empty($r['path'])) return $r['path'];
    return $upload_dir_web . $r['filename'];
};

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
/* Barra fixa + respiro para não cortar a 1ª linha */
#toolbar {
  position: sticky; top: 72px; z-index: 10;
  background: #161b22; border: 1px solid rgba(255,255,255,.08);
  border-radius: .5rem; padding: .85rem 1rem;
  box-shadow: 0 6px 14px rgba(0,0,0,.25);
}
.grid-spacer { height: .75rem; }

/* Grid leve */
.thumb-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(160px,1fr)); gap: .75rem; }
.thumb {
  background:#0f141a; border:1px solid rgba(255,255,255,.06); border-radius:.5rem; overflow:hidden;
  will-change: transform;
}
.thumb .ph{
  width:100%; aspect-ratio:4/3; background:#0b0f14; display:flex; align-items:center; justify-content:center;
  position:relative; color:#3c4452; font-size:.85rem;
}
.thumb img{
  width:100%; height:100%; object-fit:cover; display:block; opacity:0; transition:opacity .12s linear;
}
.thumb.loaded img{ opacity:1; }

/* Rodapé */
.thumb .meta{ font-size:.78rem; color:#9aa4b2; padding:.45rem .55rem; display:flex; justify-content:space-between; gap:.5rem; }

/* Seleção */
.thumb.sel{ outline:2px solid #2ea043; }
.badge-ord{
  position:absolute; top:.35rem; left:.35rem; background:#2ea043; color:#0b0f14;
  font-weight:700; font-size:.72rem; border-radius:999px; padding:.15rem .45rem;
  box-shadow:0 0 0 2px rgba(0,0,0,.35); pointer-events:none;
}

/* Botões compactos */
.btn-tight{ padding:.4rem .6rem; }
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0"><i class="bi bi-images me-2 text-primary"></i>Ordenar Fotos</h1>
    <a class="btn btn-outline-secondary" href="<?= $BASE ?>/imoveis/<?= (int)$property_id ?>/fotos">
      <i class="bi bi-arrow-left"></i> Voltar às fotos
    </a>
  </div>

  <div id="toolbar" class="mb-2">
    <div class="d-flex flex-wrap align-items-center gap-2">
      <div class="me-2 fw-semibold"><i class="bi bi-list-ol me-2"></i>Ordenar por cliques</div>

      <div class="form-check form-switch me-2">
        <input class="form-check-input" type="checkbox" id="ckComplete" checked>
        <label class="form-check-label small" for="ckComplete">Completar restantes automaticamente</label>
      </div>

      <button id="btnClear" class="btn btn-outline-warning btn-tight">
        <i class="bi bi-eraser"></i> Limpar
      </button>

      <button id="btnSave" class="btn btn-success btn-tight">
        <i class="bi bi-check2"></i> <span id="saveTxt">Salvar ordem (0/<?= count($rows) ?>)</span>
      </button>

      <div class="ms-auto small text-muted">
        Clique nas miniaturas para definir a sequência. As não selecionadas serão anexadas ao final (se ativado).
      </div>
    </div>
  </div>

  <div class="grid-spacer"></div>

  <div id="grid" class="thumb-grid">
    <?php foreach ($rows as $r): ?>
      <div class="thumb" data-id="<?= (int)$r['id'] ?>">
        <div class="ph">
          <img data-src="<?= htmlspecialchars($thumb($r)) ?>" alt="Foto <?= (int)$r['id'] ?>" loading="lazy" decoding="async" fetchpriority="low" width="400" height="300">
          <span class="badge-ord d-none">1</span>
        </div>
        <div class="meta">
          <span>Atual: #<?= (int)$r['ordem'] ?></span>
          <span>ID <?= (int)$r['id'] ?></span>
        </div>
      </div>
    <?php endforeach; ?>
  </div>
</div>

<script>
(() => {
  const grid       = document.getElementById('grid');
  const btnSave    = document.getElementById('btnSave');
  const btnClear   = document.getElementById('btnClear');
  const saveTxt    = document.getElementById('saveTxt');
  const ckComplete = document.getElementById('ckComplete');

  // Cache: id -> elemento (lookup O(1))
  const thumbs = Array.from(grid.querySelectorAll('.thumb'));
  const byId   = new Map(thumbs.map(el => [parseInt(el.dataset.id, 10), el]));
  const total  = thumbs.length;

  // Estado de seleção
  const picked    = [];        // sequência (array)
  const pickedSet = new Set(); // lookup rápido

  function mark(el, posNumber){
    el.classList.add('sel');
    const b = el.querySelector('.badge-ord');
    b.textContent = posNumber;
    b.classList.remove('d-none');
  }
  function unmark(el){
    el.classList.remove('sel');
    el.querySelector('.badge-ord').classList.add('d-none');
  }

  function updateCounter(){
    saveTxt.textContent = `Salvar ordem (${picked.length}/${total})`;
  }

  // Clique numa thumb: adiciona/remove sem varrer a grid
  grid.addEventListener('click', (ev) => {
    const t = ev.target.closest('.thumb');
    if (!t) return;
    const id = parseInt(t.dataset.id, 10);

    if (pickedSet.has(id)) {
      const idx = picked.indexOf(id);
      if (idx >= 0) picked.splice(idx, 1);
      pickedSet.delete(id);
      unmark(t);
      // reenumera só quem vem depois
      for (let i = idx; i < picked.length; i++) {
        const el = byId.get(picked[i]);
        if (el) { el.querySelector('.badge-ord').textContent = i + 1; }
      }
    } else {
      picked.push(id);
      pickedSet.add(id);
      mark(t, picked.length);
    }
    updateCounter();
  }, { passive: true });

  // Limpar seleção
  btnClear.addEventListener('click', () => {
    for (const id of picked) {
      const el = byId.get(id);
      if (el) unmark(el);
    }
    picked.length = 0; pickedSet.clear();
    updateCounter();
  });

  // Salvar
  btnSave.addEventListener('click', async () => {
    if (picked.length === 0 && !ckComplete.checked) {
      alert('Selecione ao menos uma foto, ou ative "Completar restantes automaticamente".');
      return;
    }
    btnSave.disabled = true; btnSave.classList.add('disabled'); saveTxt.textContent = 'Salvando...';

    const form = new FormData();
    form.append('order', picked.join(','));
    form.append('complete_rest', ckComplete.checked ? '1' : '0');

    try {
      await fetch(location.href, { method: 'POST', body: form, credentials: 'same-origin' });
      // Redireciona para a tela de fotos
      window.location.href = "<?= $BASE ?>/imoveis/<?= (int)$property_id ?>/fotos?ok=ordem";
    } catch (e) {
      alert('Falha ao salvar. Tente novamente.');
      btnSave.disabled = false; btnSave.classList.remove('disabled');
      updateCounter();
    }
  });

  // Lazy-load leve com IntersectionObserver (rootMargin generoso)
  const io = new IntersectionObserver((entries, obs) => {
    for (const e of entries) {
      if (!e.isIntersecting) continue;
      const img = e.target;
      img.src = img.dataset.src;
      img.addEventListener('load', () => img.closest('.thumb').classList.add('loaded'), { once: true });
      obs.unobserve(img);
    }
  }, { rootMargin: '600px 0px' });

  document.querySelectorAll('#grid img[data-src]').forEach(img => io.observe(img));
})();
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>