<?php
// Diagnóstico (desative em produção)
error_reporting(E_ALL);
ini_set('display_errors', 1);

if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('imoveis','update');

$BASE = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title = 'Fotos do Imóvel';

$property_id = (int)($_GET['id'] ?? 0);
if ($property_id <= 0) { header("Location: {$BASE}/imoveis"); exit; }

// === Carrega imóvel com fallback se 'owner_user_id' não existir ===
$imovel = null;
try {
    $imovel = fetch_one("SELECT id, titulo, owner_user_id FROM properties WHERE id = ?", [$property_id], true);
} catch (Throwable $e) {
    try {
        $imovel = fetch_one("SELECT id, titulo FROM properties WHERE id = ?", [$property_id], true);
        if ($imovel) { $imovel['owner_user_id'] = null; }
    } catch (Throwable $e2) {
        http_response_code(500);
        exit('Erro ao carregar imóvel.');
    }
}
if (!$imovel) { header("Location: {$BASE}/imoveis"); exit; }

// === Escopo de update (own/all) só se a coluna existir ===
$scopeUpd = rbac_scope('imoveis','update'); // 'none' | 'own' | 'all'
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
if (
    $scopeUpd === 'own'
    && array_key_exists('owner_user_id', $imovel)
    && isset($imovel['owner_user_id'])
    && (int)$imovel['owner_user_id'] !== $me_id
) {
    http_response_code(403);
    exit('Você não tem permissão para gerenciar as fotos deste imóvel.');
}

// Pastas
$upload_dir_fs  = __DIR__ . '/../../uploads/imoveis/' . $property_id . '/';           // filesystem
$upload_dir_web = rtrim($BASE, '/') . '/uploads/imoveis/' . $property_id . '/';       // web

// Helpers
function _parse_size_to_bytes($val) {
    $val = trim((string)$val);
    if ($val === '') return 0;
    $last = strtolower($val[strlen($val)-1]);
    $num  = (int)$val;
    switch ($last) {
        case 'g': return $num * 1024 * 1024 * 1024;
        case 'm': return $num * 1024 * 1024;
        case 'k': return $num * 1024;
        default:  return (int)$val;
    }
}
$maxPerFile = 32 * 1024 * 1024; // 32MB por arquivo (ajuste em php.ini também)
$phpMaxFile = _parse_size_to_bytes(ini_get('upload_max_filesize'));
if ($phpMaxFile > 0) $maxPerFile = min($maxPerFile, $phpMaxFile);

// Mensagens via GET
$ok  = isset($_GET['ok'])  ? $_GET['ok']  : null;
$err = isset($_GET['err']) ? $_GET['err'] : null;

// === POST: Upload / principal / excluir ===
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $isAjax = isset($_POST['ajax']) && $_POST['ajax'] === '1';
    $json   = ['uploaded'=>0,'errors'=>[]];

    // Upload (suporta 1..N fotos)
    if (isset($_FILES['fotos']) && !empty($_FILES['fotos']['name'][0])) {
        try {
            if (!is_dir($upload_dir_fs)) {
                if (!@mkdir($upload_dir_fs, 0755, true) && !is_dir($upload_dir_fs)) {
                    throw new Exception('Não foi possível criar diretório de upload.');
                }
            }

            $allowed_ext = ['jpg','jpeg','png','webp'];
            $okUp = 0; $erros = [];

            foreach ($_FILES['fotos']['name'] as $k => $name) {
                if ($_FILES['fotos']['error'][$k] !== UPLOAD_ERR_OK) { $erros[] = "Falha: {$name}"; continue; }
                $tmp  = $_FILES['fotos']['tmp_name'][$k];
                $size = (int)$_FILES['fotos']['size'][$k];
                $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

                if (!in_array($ext, $allowed_ext, true)) { $erros[] = "Extensão não permitida: {$name}"; continue; }
                if ($maxPerFile > 0 && $size > $maxPerFile) { $erros[] = "Arquivo muito grande: {$name}"; continue; }

                $filename = uniqid('', true) . '.' . $ext;
                $dest_fs  = $upload_dir_fs . $filename;
                $dest_web = $upload_dir_web . $filename;

                if (!move_uploaded_file($tmp, $dest_fs)) { $erros[] = "Erro ao mover: {$name}"; continue; }

                // Inserção em property_photos com quedas de compatibilidade
                $insert_ok = false;
                try {
                    execute(
                        "INSERT INTO property_photos (property_id, filename, path, is_main, ordem) VALUES (?, ?, ?, 0, 0)",
                        [$property_id, $filename, $dest_web]
                    );
                    $insert_ok = true;
                } catch (Throwable $e1) {
                    try {
                        execute(
                            "INSERT INTO property_photos (property_id, filename, is_main, ordem) VALUES (?, ?, 0, 0)",
                            [$property_id, $filename]
                        );
                        $insert_ok = true;
                    } catch (Throwable $e2) {
                        try {
                            execute(
                                "INSERT INTO property_photos (property_id, filename) VALUES (?, ?)",
                                [$property_id, $filename]
                            );
                            $insert_ok = true;
                        } catch (Throwable $e3) {
                            if (file_exists($dest_fs)) @unlink($dest_fs);
                            $erros[] = "Erro ao salvar no banco: {$name}";
                        }
                    }
                }

                if ($insert_ok) { $okUp++; }
            }

            if ($isAjax) {
                header('Content-Type: application/json; charset=utf-8');
                $json['uploaded'] = $okUp;
                $json['errors']   = $erros;
                echo json_encode($json);
                exit;
            } else {
                $msg = "";
                if ($okUp > 0) { $msg .= "{$okUp} foto(s) enviada(s) com sucesso!"; }
                if (!empty($erros)) { $msg .= ($msg ? ' ' : '') . implode(' | ', $erros); }
                header("Location: {$BASE}/imoveis/{$property_id}/fotos?" . ($okUp?'ok=':'err=') . urlencode($msg));
                exit;
            }
        } catch (Throwable $e) {
            if ($isAjax) {
                header('Content-Type: application/json; charset=utf-8');
                $json['errors'][] = 'Erro ao processar upload.';
                echo json_encode($json);
                exit;
            } else {
                header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=" . urlencode('Erro ao processar upload.'));
                exit;
            }
        }
    }

    // Definir principal
    if (isset($_POST['definir_principal']) && is_numeric($_POST['definir_principal'])) {
        $foto_id = (int)$_POST['definir_principal'];
        try {
            execute("UPDATE property_photos SET is_main = 0 WHERE property_id = ?", [$property_id]);
            execute("UPDATE property_photos SET is_main = 1 WHERE id = ? AND property_id = ?", [$foto_id, $property_id]);
            header("Location: {$BASE}/imoveis/{$property_id}/fotos?ok=" . urlencode('Foto principal definida.'));
            exit;
        } catch (Throwable $e) {
            header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=" . urlencode("Seu banco não possui a coluna 'is_main'."));
            exit;
        }
    }

    // Excluir foto
    if (isset($_POST['excluir_foto']) && is_numeric($_POST['excluir_foto'])) {
        $foto_id = (int)$_POST['excluir_foto'];
        try {
            $foto = fetch_one("SELECT filename, path FROM property_photos WHERE id = ? AND property_id = ?", [$foto_id, $property_id], true);
            if ($foto) {
                $file_fs = $upload_dir_fs . $foto['filename'];
                if (file_exists($file_fs)) @unlink($file_fs);
                execute("DELETE FROM property_photos WHERE id = ? AND property_id = ?", [$foto_id, $property_id]);
                header("Location: {$BASE}/imoveis/{$property_id}/fotos?ok=" . urlencode('Foto excluída.'));
                exit;
            } else {
                header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=" . urlencode('Foto não encontrada.'));
                exit;
            }
        } catch (Throwable $e) {
            header("Location: {$BASE}/imoveis/{$property_id}/fotos?err=" . urlencode('Erro ao excluir foto.'));
            exit;
        }
    }
}

// === Carregar fotos com fallback se 'path'/'ordem' não existirem ===
$fotos = [];
$hasPath = true;
try {
    $fotos = fetch_all(
        "SELECT id, filename, path, is_main, created_at, COALESCE(ordem,0) AS ordem
           FROM property_photos
          WHERE property_id = ?
       ORDER BY COALESCE(ordem,0) ASC, created_at ASC",
        [$property_id],
        true
    );
} catch (Throwable $e) {
    $hasPath = false;
    try {
        $fotos = fetch_all(
            "SELECT id, filename, is_main, created_at
               FROM property_photos
              WHERE property_id = ?
           ORDER BY id ASC",
            [$property_id],
            true
        );
    } catch (Throwable $e2) {
        $fotos = [];
    }
}

$total_fotos = is_array($fotos) ? count($fotos) : 0;
$foto_principal = null;
foreach ($fotos as $f) { if (!empty($f['is_main']) && (int)$f['is_main'] === 1) { $foto_principal = $f; break; } }

// Resolve URL da imagem
$foto_web = function(array $f) use ($upload_dir_web, $hasPath) {
    if ($hasPath && !empty($f['path'])) return $f['path'];
    return $upload_dir_web . $f['filename'];
};

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
/* Upload queue (escuro) */
.upload-wrap{ background:#0f141a; border:1px solid rgba(255,255,255,.08); border-radius:.5rem; padding:1rem; }
.file-row{ background:#0b0f14; border:1px solid rgba(255,255,255,.06); border-radius:.375rem; padding:.5rem .75rem; margin-bottom:.5rem; }
.file-row .name{ color:#c9d1d9; font-size:.9rem; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
.file-row .size{ color:#9aa4b2; font-size:.78rem; }
.progress{ height:8px; background:#0f2533; }
.progress-bar{ background:#198754; transition: width .15s linear; }
.progress-lg{ height:12px; }
.queue-head{ color:#c9d1d9; }
.queue-log{ color:#9aa4b2; font-size:.8rem; }
.badge-soft-danger{ background:rgba(220,53,69,.15); color:#f08a8f; border:1px solid rgba(220,53,69,.25); }

/* Galeria leve */
.card-img-top{ height:200px; object-fit:cover; }
.thumb-lazy{ opacity:0; transition: opacity .12s linear; }
.thumb-lazy.loaded{ opacity:1; }
</style>

<div class="container mt-4 mb-5">
  <div class="d-flex justify-content-between align-items-center mb-4">
      <div>
          <h1 class="fw-bold mb-1">
              <i class="bi bi-images text-primary me-2"></i>
              Fotos - <?= htmlspecialchars($imovel['titulo']) ?>
          </h1>
          <p class="text-muted mb-0">Gerencie as fotos do imóvel (<?= $total_fotos ?> foto<?= $total_fotos != 1 ? 's' : '' ?>)</p>
      </div>
      <div class="d-flex flex-wrap gap-2">
          <a href="<?= $BASE ?>/imoveis" class="btn btn-outline-secondary">
              <i class="bi bi-list me-2"></i>Ver Imóveis
          </a>
          <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/editar" class="btn btn-outline-primary">
              <i class="bi bi-pencil me-2"></i>Editar Imóvel
          </a>
          <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/ordenar" class="btn btn-primary">
              <i class="bi bi-list-ol me-2"></i>Ordenar Fotos
          </a>
      </div>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($ok) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>
  <?php if ($err): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($err) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <!-- Upload (AJAX com fila + progresso geral + fallback) -->
  <div class="card mb-4">
    <div class="card-header">
      <h5 class="mb-0"><i class="bi bi-cloud-upload me-2"></i>Adicionar Novas Fotos</h5>
    </div>
    <div class="card-body">
      <form id="formUpload" method="POST" enctype="multipart/form-data">
        <div class="mb-3">
          <label class="form-label">Selecionar Fotos (JPG/PNG/WEBP)</label>
          <input type="file" id="inpFotos" name="fotos[]" class="form-control" multiple accept="image/*">
          <div class="form-text">Selecione várias fotos. O envio ocorre em fila com progresso individual e total.</div>
        </div>

        <!-- Progresso geral -->
        <div id="overallWrap" class="upload-wrap d-none mb-3">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <div class="fw-semibold queue-head"><i class="bi bi-speedometer2 me-2"></i>Progresso geral</div>
            <div class="small text-muted" id="overallPct">0%</div>
          </div>
          <div class="progress progress-lg">
            <div id="overallBar" class="progress-bar" role="progressbar" style="width:0%"></div>
          </div>
          <div class="mt-2 d-flex justify-content-between queue-log">
            <div><span id="qStats">0/0</span> arquivos</div>
            <div id="qBytes">0 MB / 0 MB</div>
          </div>
        </div>

        <!-- Fila -->
        <div id="queue" class="upload-wrap d-none">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <div class="fw-semibold queue-head"><i class="bi bi-list-task me-2"></i>Fila de upload</div>
            <div class="small text-muted" id="qInfo"></div>
          </div>
          <div id="qItems"></div>
          <div class="d-flex gap-2">
            <button type="button" id="btnStart" class="btn btn-success">
              <i class="bi bi-upload me-2"></i>Iniciar envio
            </button>
            <button type="button" id="btnClearQueue" class="btn btn-outline-warning">
              <i class="bi bi-eraser me-2"></i>Limpar fila
            </button>
            <button type="button" id="btnReload" class="btn btn-outline-secondary d-none">
              <i class="bi bi-arrow-clockwise me-2"></i>Atualizar galeria
            </button>
          </div>
        </div>

        <noscript>
          <div class="mt-3">
            <button type="submit" class="btn btn-primary">
              <i class="bi bi-upload me-2"></i>Enviar Fotos
            </button>
          </div>
        </noscript>
      </form>
    </div>
  </div>

  <!-- Status Principal -->
  <div class="alert alert-info mb-4">
    <div class="d-flex align-items-center">
      <i class="bi bi-star-fill text-warning me-2 fs-5"></i>
      <div>
        <strong>Foto Principal:</strong>
        <?php if ($foto_principal): ?>
          Definida
        <?php else: ?>
          <span class="text-warning">Nenhuma foto principal definida</span>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- Galeria -->
  <?php if ($total_fotos > 0): ?>
    <div class="row g-3">
      <?php foreach ($fotos as $idx => $f): ?>
        <div class="col-md-3 col-sm-6 mb-3">
          <div class="card h-100 position-relative">
            <img
              data-src="<?= htmlspecialchars($foto_web($f)) ?>"
              class="card-img-top thumb-lazy"
              alt="Foto <?= $idx+1 ?>"
              loading="lazy"
              decoding="async"
              fetchpriority="low"
              onerror="this.src='<?= $BASE ?>/assets/img/no-image.jpg'">

            <?php if (!empty($f['is_main']) && (int)$f['is_main'] === 1): ?>
              <span class="position-absolute top-0 start-0 badge bg-warning rounded-pill m-2">
                <i class="bi bi-star-fill me-1"></i>Principal
              </span>
            <?php endif; ?>

            <div class="card-body d-flex flex-column">
              <h6 class="card-title mb-2">Foto <?= $idx+1 ?></h6>
              <small class="text-muted mb-2 flex-grow-1">
                <?php
                  if (!empty($f['created_at'])) {
                    echo date('d/m/Y H:i', strtotime($f['created_at']));
                  } elseif (isset($f['ordem'])) {
                    echo 'Ordem: #' . (int)$f['ordem'];
                  }
                ?>
              </small>

              <form method="POST" class="d-inline">
                <input type="hidden" name="definir_principal" value="<?= (int)$f['id'] ?>">
                <button type="submit" class="btn btn-sm btn-outline-warning w-100 mb-1" <?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? 'disabled' : '' ?>>
                  <i class="bi bi-star<?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? '-fill' : '' ?> me-1"></i>
                  <?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? 'Principal' : 'Definir Principal' ?>
                </button>
              </form>

              <form method="POST" class="d-inline" onsubmit="return confirm('Excluir esta foto?');">
                <input type="hidden" name="excluir_foto" value="<?= (int)$f['id'] ?>">
                <button type="submit" class="btn btn-sm btn-outline-danger w-100">
                  <i class="bi bi-trash me-1"></i>Excluir
                </button>
              </form>
            </div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
  <?php else: ?>
    <div class="text-center py-5">
      <i class="bi bi-images display-1 text-muted mb-3"></i>
      <h4 class="text-muted">Nenhuma foto cadastrada</h4>
      <p class="text-muted">Use a seção acima para adicionar fotos ao imóvel.</p>
    </div>
  <?php endif; ?>
</div>

<script>
(() => {
  // --- Lazy-load das miniaturas
  const io = new IntersectionObserver((entries, obs) => {
    for (const e of entries) {
      if (!e.isIntersecting) continue;
      const img = e.target;
      img.src = img.dataset.src;
      img.addEventListener('load', () => img.classList.add('loaded'), { once: true });
      obs.unobserve(img);
    }
  }, { rootMargin: '600px 0px' });
  document.querySelectorAll('.thumb-lazy[data-src]').forEach(img => io.observe(img));
})();

(() => {
  // --- Fila de upload com progresso geral e concorrência controlada ---
  const inp   = document.getElementById('inpFotos');
  const queue = document.getElementById('queue');
  const qItems= document.getElementById('qItems');
  const qInfo = document.getElementById('qInfo');
  const qStats= document.getElementById('qStats');
  const qBytes= document.getElementById('qBytes');
  const btnGo = document.getElementById('btnStart');
  const btnCl = document.getElementById('btnClearQueue');
  const btnRl = document.getElementById('btnReload');
  const form  = document.getElementById('formUpload');

  const overallWrap = document.getElementById('overallWrap');
  const overallBar  = document.getElementById('overallBar');
  const overallPct  = document.getElementById('overallPct');

  const MAX_PER_FILE = <?= (int)$maxPerFile ?>; // mesmo limite do servidor (bytes)
  const CONCURRENCY  = 3;                        // 3 uploads em paralelo (rápido e estável)

  let files = [];
  let rows  = new Map(); // index -> dom row
  let sent  = 0;
  let errors= 0;

  // controle do progresso total
  let totalBytes = 0;
  let uploadedBytes = 0;
  let lastLoadedPerIndex = {}; // index -> last loaded for delta

  function fmtSize(b){
    if (b >= 1024*1024) return (b/1024/1024).toFixed(1)+' MB';
    if (b >= 1024)      return (b/1024).toFixed(1)+' KB';
    return b + ' B';
  }

  function renderQueue() {
    qItems.innerHTML = '';
    rows.clear();
    totalBytes = files.reduce((a,f) => a + f.size, 0);
    uploadedBytes = 0;
    lastLoadedPerIndex = {};
    qStats.textContent = `0/${files.length}`;
    qBytes.textContent = `0 MB / ${(totalBytes/1024/1024).toFixed(1)} MB`;
    overallBar.style.width = '0%';
    overallPct.textContent = '0%';

    files.forEach((f, i) => {
      const row = document.createElement('div');
      row.className = 'file-row';
      row.innerHTML = `
        <div class="d-flex justify-content-between align-items-center">
          <div class="name" title="${f.name}">${f.name}</div>
          <div class="size">${fmtSize(f.size)}</div>
        </div>
        <div class="progress mt-2">
          <div class="progress-bar" role="progressbar" style="width:0%"></div>
        </div>
        <div class="mt-1 queue-log d-none"></div>
      `;
      row.dataset.index = i;
      qItems.appendChild(row);
      rows.set(i, row);
    });

    queue.classList.toggle('d-none', files.length === 0);
    overallWrap.classList.toggle('d-none', files.length === 0);
    qInfo.textContent = `Fila pronta`;
  }

  inp.addEventListener('change', () => {
    const picked = Array.from(inp.files || []);

    // valida tamanho no cliente
    const oversized = picked.filter(f => MAX_PER_FILE > 0 && f.size > MAX_PER_FILE);
    const ok = picked.filter(f => !(MAX_PER_FILE > 0 && f.size > MAX_PER_FILE));

    files = ok;
    sent  = 0;
    errors= 0;

    renderQueue();

    if (oversized.length) {
      const msg = `Arquivos ignorados por tamanho (> ${Math.round(MAX_PER_FILE/1024/1024)}MB): ` +
                  oversized.map(f => f.name).slice(0,3).join(', ') +
                 (oversized.length > 3 ? ` e +${oversized.length-3}…` : '');
      qInfo.innerHTML = `<span class="badge badge-soft-danger">${msg}</span>`;
    }
  });

  btnCl.addEventListener('click', () => {
    files = [];
    sent  = 0;
    errors= 0;
    inp.value = '';
    renderQueue();
  });

  btnRl.addEventListener('click', () => location.reload());

  // atualiza barra geral (delta baseado no progresso do arquivo)
  function updateOverall(index, loaded, total) {
    const last = lastLoadedPerIndex[index] || 0;
    const delta = Math.max(0, loaded - last);
    lastLoadedPerIndex[index] = loaded;
    uploadedBytes += delta;

    const pct = totalBytes > 0 ? Math.min(100, Math.round((uploadedBytes / totalBytes) * 100)) : 0;
    overallBar.style.width = pct + '%';
    overallPct.textContent = pct + '%';
    qBytes.textContent = `${(uploadedBytes/1024/1024).toFixed(1)} MB / ${(totalBytes/1024/1024).toFixed(1)} MB`;
  }

  // worker de upload (um arquivo por chamada)
  function uploadOne(i) {
    return new Promise((resolve) => {
      const f   = files[i];
      const row = rows.get(i);
      const bar = row.querySelector('.progress-bar');
      const log = row.querySelector('.queue-log');

      const fd = new FormData();
      fd.append('ajax','1');
      fd.append('fotos[]', f, f.name);

      const xhr = new XMLHttpRequest();
      xhr.open('POST', window.location.href, true);

      xhr.upload.onprogress = (ev) => {
        if (ev.lengthComputable) {
          const pct = Math.max(1, Math.round(ev.loaded * 100 / ev.total));
          bar.style.width = pct + '%';
          updateOverall(i, ev.loaded, ev.total);
        }
      };

      xhr.onreadystatechange = () => {
        if (xhr.readyState === 4) {
          let okResp = false;
          if (xhr.status >= 200 && xhr.status < 300) {
            try {
              const resp = JSON.parse(xhr.responseText || '{}');
              if (resp && typeof resp.uploaded !== 'undefined') {
                if (resp.uploaded === 1) {
                  okResp = true;
                } else {
                  errors++;
                  log.classList.remove('d-none');
                  log.textContent = (resp.errors && resp.errors.length)
                    ? resp.errors.join(' | ')
                    : 'Falha no upload.';
                }
              } else {
                // resposta não-JSON (ex.: página de erro do servidor)
                errors++;
                log.classList.remove('d-none');
                log.textContent = 'Resposta inesperada do servidor.';
              }
            } catch(e) {
              errors++;
              log.classList.remove('d-none');
              log.textContent = 'Falha ao ler resposta do servidor.';
            }
          } else {
            errors++;
            log.classList.remove('d-none');
            log.textContent = 'Erro HTTP ' + xhr.status;
          }

          if (okResp) {
            bar.style.width = '100%';
          } else {
            bar.classList.remove('bg-success');
            bar.classList.add('bg-danger');
          }

          sent++;
          qStats.textContent = `${sent}/${files.length}`;
          resolve();
        }
      };

      xhr.send(fd);
    });
  }

  // agendador de concorrência
  async function runQueue() {
    if (files.length === 0) return;

    btnGo.disabled = true;
    btnCl.disabled = true;
    inp.disabled   = true;
    btnRl.classList.add('d-none');

    qInfo.textContent = `Enviando… (até ${CONCURRENCY} em paralelo)`;

    const indexes = files.map((_,i) => i);
    let active = 0, ptr = 0;

    await new Promise((done) => {
      function spawn() {
        while (active < CONCURRENCY && ptr < indexes.length) {
          active++;
          const i = indexes[ptr++];
          uploadOne(i).then(() => {
            active--;
            if (ptr < indexes.length) {
              spawn();
            } else if (active === 0) {
              done();
            }
          });
        }
      }
      spawn();
    });

    // fim
    qInfo.textContent = errors === 0
      ? 'Tudo certo! Atualizando a galeria…'
      : `Concluído com ${errors} erro(s). Verifique os itens marcados.`;

    if (errors === 0) {
      // pequeno delay apenas para o usuário ver 100%
      setTimeout(() => location.reload(), 600);
    } else {
      btnRl.classList.remove('d-none');
      btnGo.disabled = false;
      btnCl.disabled = false;
      inp.disabled   = false;
    }
  }

  btnGo.addEventListener('click', runQueue);

  // Fallback: submit normal não interceptado
  form.addEventListener('submit', () => {});
})();
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>