<?php
/**
 * /modules/cron/alertas_inatividade.php
 * Envia alertas AUTOMÁTICOS de inatividade por e-mail.
 * Execute via CRON (CLI):
 *   php /caminho/para/troyacrm/modules/cron/alertas_inatividade.php
 *
 * Sugestão em cPanel (PHP 8.1):
 *   /usr/local/bin/ea-php81 /home/SEU_USUARIO/public_html/troyacrm/modules/cron/alertas_inatividade.php >/dev/null 2>&1
 */

if (php_sapi_name() !== 'cli') { http_response_code(403); exit("CLI only\n"); }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/mailer.php';

/* -----------------------------------------------------------
   Helpers locais (nomes únicos para evitar conflitos)
----------------------------------------------------------- */

/** Lê configuração em settings (com default) */
function _cfg_get_cron($key, $default=null){
  try {
    $r = fetch_one("SELECT value FROM settings WHERE `key` = ? LIMIT 1", [$key], true);
    if ($r && array_key_exists('value', $r)) return $r['value'];
  } catch (Throwable $e) {}
  return $default;
}

/** Template de e-mail (AUTOMÁTICO), igual ao manual (só muda o rodapé) */
function _alerts_build_inactivity_email_auto(array $data, string $base): array {
  $id       = (int)($data['id'] ?? 0);
  $cliente  = (string)($data['cliente_nome'] ?? '—');
  $status   = (string)($data['status'] ?? '—');
  $lastRef  = $data['last_dt'] ?? null;

  $gapDays = 0;
  if ($lastRef) {
    try {
      $diff    = (new DateTime())->diff(new DateTime($lastRef));
      $gapDays = (int)$diff->format('%a');
    } catch (Throwable $e) { $gapDays = 0; }
  }

  $subject = "Alerta de inatividade • Atendimento #{$id} — {$cliente}";
  $link    = rtrim($base, '/') . '/atendimentos/visualizar?id=' . $id;

  $html = "
    <div style='font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;font-size:14px;color:#111'>
      <h2 style='margin:0 0 .6rem'>Alerta de inatividade</h2>

      <p style='margin:.4rem 0'>Atendimento <strong>#{$id}</strong> do cliente
        <strong>".htmlspecialchars($cliente)."</strong>.</p>

      <p style='margin:.4rem 0'>
        Última atividade: <strong>".($lastRef ? date('d/m/Y H:i', strtotime($lastRef)) : '—')."</strong>
        (≈ <strong>{$gapDays} dia(s)</strong> sem atualização).
      </p>

      <table cellpadding='6' cellspacing='0' style='border:1px solid #e5e7eb;border-radius:8px;margin:.6rem 0'>
        <tr>
          <td><strong>Status</strong></td>
          <td>".htmlspecialchars(ucfirst(str_replace('_',' ', $status)))."</td>
        </tr>
        <tr>
          <td><strong>Cliente</strong></td>
          <td>".htmlspecialchars($cliente)."</td>
        </tr>
        <tr>
          <td><strong>ID</strong></td>
          <td>#{$id}</td>
        </tr>
      </table>

      <p style='margin:.8rem 0 .2rem; font-weight:600'>Por favor, atualize este atendimento para não perder o cliente:</p>
      <ul style='margin:.2rem 0 .8rem .95rem; padding:0'>
        <li>Registre uma nova interação (ligação, WhatsApp, e-mail ou visita);</li>
        <li>Agende a próxima ação com data/hora;</li>
        <li>Se necessário, reatribua para outro responsável.</li>
      </ul>

      <p style='margin:1rem 0'>
        <a href='{$link}' style='background:#198754;color:#fff;padding:.55rem .9rem;border-radius:.45rem;text-decoration:none;display:inline-block'>
          Abrir atendimento
        </a>
      </p>

      <hr style='border:none;border-top:1px solid #ddd;margin:1rem 0'>
      <p style='color:#555'>Mensagem automática do Troya CRM.</p>
    </div>
  ";

  return ['subject' => $subject, 'html' => $html];
}

/** Resolve destinatários a partir dos targets (sempre inclui o corretor/owner) */
function _recipients_from_targets(array $row, string $targetsCsv): array {
  $targets = array_filter(array_map('trim', explode(',', (string)$targetsCsv)));

  $list = [];
  // Sempre inclui o corretor (owner) no automático:
  if (!empty($row['owner_email'])) {
    $list[] = ['email'=>$row['owner_email'], 'nome'=>($row['owner_nome'] ?: 'Corretor')];
  }

  // Admins / Gerentes (no banco a role de gerente costuma ser "manager")
  $rolesWanted = [];
  if (in_array('admin', $targets, true))   $rolesWanted[] = 'admin';
  if (in_array('gerente', $targets, true)) $rolesWanted[] = 'manager';

  if ($rolesWanted) {
    $place = implode(',', array_fill(0, count($rolesWanted), '?'));
    $users = fetch_all("SELECT nome, email FROM users WHERE role IN ($place) AND email IS NOT NULL AND email <> ''", $rolesWanted, true) ?: [];
    foreach ($users as $u) { $list[] = ['email'=>$u['email'], 'nome'=>$u['nome']]; }
  }

  // de-dup por e-mail
  $uniq=[]; $out=[];
  foreach ($list as $r){ $e=strtolower(trim($r['email'] ?? '')); if($e && !isset($uniq[$e])){ $uniq[$e]=1; $out[]=$r; } }
  return $out;
}

/* -----------------------------------------------------------
   Settings
----------------------------------------------------------- */
$enabled = _cfg_get_cron('alerts.enabled', '1');
$days    = (int)_cfg_get_cron('alerts.days_without_update', '3');
$targets = (string)_cfg_get_cron('alerts.target_roles', 'admin,gerente'); // sem 'corretor' aqui; owner já é incluído sempre

if ($enabled !== '1') { echo "alerts.enabled=0; abortado.\n"; exit; }
if ($days < 1) $days = 1;

/* Base (link do e-mail)
   - Ideal: BASE_URL absoluto definido em config (ex.: https://dominio.com/troyacrm)
   - Fallback: caminho relativo /troyacrm (ajuste se necessário) */
$BASE = defined('BASE_URL') ? rtrim(BASE_URL, '/') : '/troyacrm';

/* -----------------------------------------------------------
   Seleção dos atendimentos parados (respeita soneca)
----------------------------------------------------------- */
$sql = "
  SELECT 
    i.id, i.status, i.alerts_snooze_until,
    COALESCE(
      (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
      i.last_activity_at, i.updated_at, i.created_at
    ) AS last_dt,
    c.nome_completo AS cliente_nome,
    u.nome  AS owner_nome,
    u.email AS owner_email
  FROM interactions i
  LEFT JOIN clients c ON c.id = i.client_id
  LEFT JOIN users   u ON u.id = i.owner_user_id
  WHERE i.status <> 'finalizado'
    AND (i.alerts_snooze_until IS NULL OR i.alerts_snooze_until < NOW())
    AND TIMESTAMPDIFF(
          DAY,
          COALESCE(
            (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
            i.last_activity_at, i.updated_at, i.created_at
          ),
          NOW()
        ) >= ?
  ORDER BY i.id ASC
  LIMIT 1000
";
$rows = fetch_all($sql, [$days], true) ?: [];

/* -----------------------------------------------------------
   Envio
----------------------------------------------------------- */
$total   = count($rows);
$sentSum = 0;   // total de destinatários enviados (soma)
$okRows  = 0;   // atendimentos com envio >= 1
$skipRows= 0;   // atendimentos ignorados (sem destinatário)

foreach ($rows as $r) {
  $recipients = _recipients_from_targets($r, $targets);
  if (!$recipients) { $skipRows++; continue; }

  $tpl = _alerts_build_inactivity_email_auto($r, $BASE);
  $ok  = mailer_send_many($recipients, $tpl['subject'], $tpl['html']);

  if ($ok > 0) {
    $okRows++;
    $sentSum += $ok;
    try {
      execute("INSERT INTO system_logs (tipo, ref_id, mensagem, created_at) VALUES ('alerta_inatividade_auto', ?, ?, NOW())",
        [$r['id'], "Enviado para {$ok} destinatário(s)"]);
    } catch (Throwable $e) { /* tabela opcional; ignore */ }
  } else {
    $skipRows++;
  }
}

echo "CRON OK: base={$total}, atendimentos_enviados={$okRows}, destinatarios_total={$sentSum}, ignorados={$skipRows}\n";