<?php
// modules/atendimentos/notificar.php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';
require_once __DIR__ . '/../../app/mailer.php';

rbac_require('atendimentos','update'); // precisa poder acionar a notificação

$BASE   = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id  = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$id     = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
if ($id <= 0) { http_response_code(400); exit('ID inválido'); }

/* Helper local p/ settings (com fallback) */
if (!function_exists('settings_get')) {
  function settings_get($key, $default=null) {
    try {
      $r = fetch_one("SELECT value FROM settings WHERE `key`=? LIMIT 1", [$key], true);
      if ($r && array_key_exists('value', $r)) return $r['value'];
    } catch(Throwable $e) {}
    try {
      $r = fetch_one("SELECT `value` FROM system_configs WHERE `key`=? LIMIT 1", [$key], true);
      if ($r && array_key_exists('value', $r)) return $r['value'];
    } catch(Throwable $e) {}
    return $default;
  }
}

/* Carrega atendimento + cliente + dono */
$row = fetch_one("
  SELECT i.*,
         c.nome_completo         AS cliente_nome,
         u.nome                  AS owner_nome,
         u.email                 AS owner_email,
         u.id                    AS owner_id
    FROM interactions i
    LEFT JOIN clients c ON c.id = i.client_id
    LEFT JOIN users   u ON u.id = i.owner_user_id
   WHERE i.id = ?
   LIMIT 1
", [$id], true);

if (!$row) { http_response_code(404); exit('Atendimento não encontrado'); }

/* RBAC (escopo own) */
$scope = rbac_scope('atendimentos','update'); // 'all'|'own'|'none'
if ($scope === 'own') {
  $owner   = (int)($row['owner_user_id'] ?? 0);
  $creator = (int)($row['created_by'] ?? 0);
  if (!in_array($me_id, [$owner, $creator], true)) {
    http_response_code(403); exit('Sem permissão neste atendimento.');
  }
}

/* Última atividade e gap */
$lastRef = $row['last_activity_at'] ?: ($row['updated_at'] ?? $row['created_at']);
$diffDays = 0;
if ($lastRef) {
  $diff = (new DateTime())->diff(new DateTime($lastRef));
  $diffDays = (int)$diff->format('%a');
}

/* Última nota (se existir) */
$lastNote = fetch_one("
  SELECT nota, created_at
    FROM interaction_logs
   WHERE interaction_id = ?
   ORDER BY id DESC
   LIMIT 1
", [$id], true);

$lastNoteText = trim((string)($lastNote['nota'] ?? ''));
$lastNoteWhen = !empty($lastNote['created_at']) ? date('d/m/Y H:i', strtotime($lastNote['created_at'])) : null;

/* Destinatários (manual SEMPRE tenta o corretor; admins/gerentes conforme config) */
$recipients = [];

// 1) Corretor
if (!empty($row['owner_email'])) {
  $recipients[] = ['email' => $row['owner_email'], 'nome' => ($row['owner_nome'] ?: 'Corretor')];
}

// 2) Admin / Gerente (conforme alerts.target_roles; mapeia 'gerente'->'manager')
$targetsStr  = (string)settings_get('alerts.target_roles', 'admin,gerente');
$targets     = array_filter(array_map('trim', explode(',', $targetsStr)));
$rolesWanted = [];
foreach ($targets as $t) {
  if ($t === 'admin')   $rolesWanted[] = 'admin';
  if ($t === 'gerente') $rolesWanted[] = 'manager';
}
if ($rolesWanted) {
  $place = implode(',', array_fill(0, count($rolesWanted), '?'));
  $extra = fetch_all("SELECT nome, email FROM users WHERE role IN ($place) AND email IS NOT NULL AND email <> ''", $rolesWanted, true) ?: [];
  foreach ($extra as $u) {
    $recipients[] = ['email' => $u['email'], 'nome' => $u['nome']];
  }
}

/* De-dup por e-mail */
$seen = []; $clean = [];
foreach ($recipients as $r) {
  $e = strtolower(trim($r['email'] ?? ''));
  if ($e !== '' && !isset($seen[$e])) { $seen[$e] = true; $clean[] = $r; }
}
$recipients = $clean;

/* Sem destinatários -> volta com erro */
$back = "{$BASE}/admin/alertas";
if (!$recipients) {
  header("Location: {$back}?err=" . urlencode('Sem destinatários com e-mail válido (corretor/admin/gerente).'));
  exit;
}

/* E-mail */
$cliente   = $row['cliente_nome'] ?: '—';
$statusRaw = (string)($row['status'] ?? '');
$statusFmt = $statusRaw !== '' ? ucfirst(str_replace('_',' ',$statusRaw)) : '—';
$assunto   = "Alerta de inatividade • Atendimento #{$row['id']} — {$cliente}";
$link      = rtrim($BASE, '/') . '/atendimentos/visualizar?id=' . (int)$row['id'];

$notaHtml = '';
if ($lastNoteText !== '') {
  $notaHtml = "
    <div style='margin:.6rem 0; padding:.6rem .75rem; background:#f6f7f9; border:1px solid #e6e7ea; border-radius:.45rem'>
      <div style='font-weight:600; margin-bottom:.25rem'>Última nota".($lastNoteWhen ? " <span style='color:#6c757d'>({$lastNoteWhen})</span>" : "").":</div>
      <div style='white-space:pre-wrap;'>".htmlspecialchars($lastNoteText)."</div>
    </div>
  ";
}

$html = "
  <div style='font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;font-size:14px;color:#111'>
    <h2 style='margin:0 0 .6rem'>Alerta de inatividade</h2>

    <p style='margin:.35rem 0'>Atendimento <strong>#{$row['id']}</strong> do cliente <strong>".htmlspecialchars($cliente)."</strong>.</p>
    <p style='margin:.35rem 0'><strong>Status:</strong> {$statusFmt} &nbsp; • &nbsp; <strong>Dias sem atividade:</strong> {$diffDays}</p>
    <p style='margin:.35rem 0'>Última atividade registrada: <strong>".($lastRef ? date('d/m/Y H:i', strtotime($lastRef)) : '—')."</strong>.</p>

    {$notaHtml}

    <p style='margin:.5rem 0'><em>Para evitar perda do interesse do cliente, por favor, atualize este atendimento ainda hoje.</em></p>

    <p style='margin:1rem 0'>
      <a href='{$link}' style='background:#198754;color:#fff;padding:.55rem .9rem;border-radius:.45rem;text-decoration:none;display:inline-block'>Abrir atendimento</a>
    </p>

    <hr style='border:none;border-top:1px solid #ddd;margin:1rem 0'>
    <p style='color:#555'>Mensagem enviada manualmente pelo Troya CRM.</p>
  </div>
";

/* Envio */
$enviados = 0;
try {
  $enviados = mailer_send_many($recipients, $assunto, $html);
} catch (Throwable $e) {
  header("Location: {$back}?err=" . urlencode('Falha no envio: '.$e->getMessage()));
  exit;
}

/* Log leve (se existir a tabela) */
if ($enviados > 0) {
  try {
    execute(
      "INSERT INTO system_logs (tipo, ref_id, mensagem, created_at) VALUES ('alerta_inatividade_manual', ?, ?, NOW())",
      [$row['id'], "Enviado para {$enviados} destinatário(s)"]
    );
  } catch (Throwable $e) { /* ok ignorar */ }
  header("Location: {$back}?ok=" . urlencode("Alerta enviado para {$enviados} destinatário(s)."));
} else {
  header("Location: {$back}?err=" . urlencode("Não foi possível enviar o alerta."));
}
exit;