<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) {
  $req = $_SERVER['REQUEST_URI'] ?? '/troyacrm/dashboard';
  header('Location: /troyacrm/login?next=' . urlencode($req));
  exit;
}

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Atendimentos';
$active     = 'atendimentos';

rbac_require('atendimentos','view');

$BASE   = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id  = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$scopeV = rbac_scope('atendimentos','view'); // 'own' | 'all'

/* ===== Helpers ===== */
if (!function_exists('format_date')) {
  function format_date($dt,$fmt='d/m/Y H:i'){ return $dt ? date($fmt, strtotime($dt)) : '-'; }
}
function badgeStatus($s){
  return [
    'novo'            => 'secondary',
    'em_atendimento'  => 'info',
    'negociacao'      => 'warning',
    'finalizado'      => 'success',
  ][$s] ?? 'secondary';
}

/* ===== Filtros ===== */
$status_all = [
  'novo'            => 'Novo',
  'em_atendimento'  => 'Em Atendimento',
  'negociacao'      => 'Negociação',
  'finalizado'      => 'Finalizado'
];
$q   = trim($_GET['q'] ?? '');
$de  = $_GET['de']  ?? '';
$ate = $_GET['ate'] ?? '';
$sel_status = array_values(array_intersect((array)($_GET['status'] ?? []), array_keys($status_all)));

/* ===== Filtro por corretor (owner) ===== */
$role = strtolower((string)($_SESSION['user']['role'] ?? ''));
$isBoss = rbac_is_admin() || in_array($role, ['manager','gerente','gestor']); // admin/gerente podem ver todos
$owner_id = isset($_GET['owner']) ? (int)$_GET['owner'] : 0;
if (!$isBoss) { $owner_id = $me_id; } // usuário comum só filtra por ele mesmo

/* Carrega lista de usuários para o select quando for admin/gerente */
$usuarios = [];
if ($isBoss) {
  try { $usuarios = fetch_all("SELECT id, nome FROM users ORDER BY nome ASC", [], true) ?: []; }
  catch (Throwable $e) { $usuarios = []; }
}

/* ===== WHERE ===== */
$where  = [];
$params = [];

if ($scopeV === 'own') {
  $where[]  = "(i.owner_user_id = ? OR (i.owner_user_id IS NULL AND i.created_by = ?))";
  $params[] = $me_id;
  $params[] = $me_id;
}

/* Aplicar filtro por corretor se informado (admin/gerente) ou forçado (usuário comum) */
if ($owner_id > 0) {
  $where[]  = "(i.owner_user_id = ? OR (i.owner_user_id IS NULL AND i.created_by = ?))";
  $params[] = $owner_id;
  $params[] = $owner_id;
}

if ($q !== '') {
  $where[]  = "(c.nome_completo LIKE ? OR i.nota LIKE ?)";
  $params[] = "%{$q}%";
  $params[] = "%{$q}%";
}

/* -> usa última atividade para filtros de data */
if ($de !== '')  { $where[] = "COALESCE(i.last_activity_at, i.updated_at, i.created_at) >= ?"; $params[] = $de.' 00:00:00'; }
if ($ate !== '') { $where[] = "COALESCE(i.last_activity_at, i.updated_at, i.created_at) <= ?"; $params[] = $ate.' 23:59:59'; }

if (!empty($sel_status) && count($sel_status) < count($status_all)) {
  $place = implode(',', array_fill(0, count($sel_status), '?'));
  $where[] = "i.status IN ($place)";
  $params  = array_merge($params, $sel_status);
}

$sqlWhere = $where ? ('WHERE '.implode(' AND ', $where)) : '';

/* ===== KPIs ===== */
$kpis = fetch_all("
  SELECT i.status, COUNT(*) qtd
  FROM interactions i
  LEFT JOIN clients c ON c.id = i.client_id
  $sqlWhere
  GROUP BY i.status
", $params, true);
$kpiMap = [];
foreach ($kpis as $k) { $kpiMap[$k['status']] = (int)$k['qtd']; }

/* ===== Listagem: última timeline de cada atendimento ===== */
$rows = fetch_all("
  SELECT 
    i.*,
    c.nome_completo AS cliente_nome,
    u.nome          AS owner_nome,
    lg.nota         AS last_nota,
    lg.status       AS last_status,
    lg.created_at   AS last_log_at,
    /* campo unificado para exibição/ordenacao */
    COALESCE(i.last_activity_at, i.updated_at, i.created_at) AS updated_ref
  FROM interactions i
  LEFT JOIN clients c ON c.id = i.client_id
  LEFT JOIN users   u ON u.id = i.owner_user_id
  LEFT JOIN (
    SELECT t1.*
    FROM interaction_logs t1
    INNER JOIN (
      SELECT interaction_id, MAX(id) AS max_id
      FROM interaction_logs
      GROUP BY interaction_id
    ) t2 ON t2.interaction_id = t1.interaction_id AND t2.max_id = t1.id
  ) lg ON lg.interaction_id = i.id
  $sqlWhere
  ORDER BY COALESCE(i.last_activity_at, i.updated_at, i.created_at) DESC, i.id DESC
  LIMIT 300
", $params, true);

/* ===== RBAC por linha ===== */
function canRow($action, $row, $me_id){
  if (!rbac_can('atendimentos', $action)) return false;
  $scope = rbac_scope('atendimentos', $action); // own|all
  if ($scope === 'all') return true;
  $owner   = (int)($row['owner_user_id'] ?? 0);
  $creator = (int)($row['created_by'] ?? 0);
  return in_array($me_id, [$owner, $creator], true);
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
/* Layout enxuto e proporcional */
.toolbar { gap:.5rem; }
.toolbar .form-control, .toolbar .form-select, .toolbar .btn { height: 38px; }
.toolbar .btn { white-space: nowrap; }

/* KPIs: número à direita do rótulo, alinhado */
.kpi{
  padding:.6rem .8rem;
  border:1px solid rgba(255,255,255,.08);
  border-radius:.5rem;
  background:#111217;
  display:flex;
  align-items:center;
  justify-content:space-between;
  min-width:170px;
  gap:.75rem;
}
.kpi .label{ color:#adb5bd; font-size:.95rem; margin:0; }
.kpi .value{
  font-weight:700; font-size:1.05rem; font-variant-numeric:tabular-nums;
  background:rgba(255,255,255,.08); padding:.1rem .55rem; border-radius:.45rem; line-height:1;
}

/* Modal filtros */
.modal .form-check { margin-bottom:.35rem; }
.badge-count { font-variant-numeric: tabular-nums; }

/* Tabela */
.table td .small-muted { color:#adb5bd; font-size:.85rem; }
.text-truncate-2 { display:-webkit-box; -webkit-line-clamp:2; -webkit-box-orient:vertical; overflow:hidden; }
</style>

<div class="container my-4">
  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center mb-3">
    <div class="d-flex align-items-center gap-2">
      <i class="bi bi-people" style="font-size:1.5rem; color: var(--primary);"></i>
      <div>
        <h1 class="h4 m-0">Atendimentos</h1>
        <div class="text-muted small"><?= $scopeV==='all' ? 'Visão de equipe' : 'Meus atendimentos' ?></div>
      </div>
    </div>
    <div class="d-flex gap-2">
      <?php if (rbac_can('atendimentos','create')): ?>
      <a href="<?=$BASE?>/atendimentos/novo" class="btn btn-primary">
        <i class="bi bi-plus-lg me-1"></i>Novo
      </a>
      <?php endif; ?>
    </div>
  </div>

  <!-- Toolbar de filtros -->
  <form class="card card-body mb-3" method="get">
    <input type="hidden" name="tab" value="<?=htmlspecialchars($_GET['tab'] ?? '')?>">
    <div class="row row-cols-1 row-cols-md-auto g-2 align-items-end toolbar">
      <div class="col">
        <label class="form-label">Busca</label>
        <input type="text" class="form-control" name="q" value="<?=htmlspecialchars($q)?>" placeholder="Cliente ou nota">
      </div>
      <div class="col">
        <label class="form-label">De</label>
        <input type="date" class="form-control" name="de" value="<?=htmlspecialchars($de)?>">
      </div>
      <div class="col">
        <label class="form-label">Até</label>
        <input type="date" class="form-control" name="ate" value="<?=htmlspecialchars($ate)?>">
      </div>

      <!-- Filtro por Corretor -->
      <div class="col">
        <label class="form-label">Corretor</label>
        <?php if ($isBoss): ?>
          <select name="owner" class="form-select">
            <option value="0">Todos</option>
            <?php foreach ($usuarios as $u): $uid=(int)$u['id']; ?>
              <option value="<?=$uid?>" <?= ($owner_id === $uid ? 'selected' : '') ?>>
                <?= htmlspecialchars($u['nome'] ?? ('Usuário #'.$uid)) ?>
              </option>
            <?php endforeach; ?>
          </select>
        <?php else: ?>
          <input type="text" class="form-control" value="<?= htmlspecialchars($_SESSION['user']['nome'] ?? 'Eu') ?>" disabled>
          <input type="hidden" name="owner" value="<?= (int)$me_id ?>">
        <?php endif; ?>
      </div>

      <!-- Botão Status -->
      <div class="col">
        <label class="form-label d-none d-md-block">&nbsp;</label>
        <button type="button" class="btn btn-outline-light" data-bs-toggle="modal" data-bs-target="#statusModal">
          <i class="bi bi-sliders me-1"></i>Status
          <span id="badgeCount" class="badge bg-secondary ms-1 badge-count"></span>
        </button>
      </div>

      <!-- Botão Lupa (submit) -->
      <div class="col">
        <label class="form-label d-none d-md-block">&nbsp;</label>
        <button class="btn btn-outline-primary" type="submit" title="Pesquisar">
          <i class="bi bi-search"></i>
        </button>
      </div>
    </div>

    <!-- Modal de status -->
    <div class="modal fade" id="statusModal" tabindex="-1" aria-hidden="true">
      <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content bg-dark border">
          <div class="modal-header">
            <h5 class="modal-title"><i class="bi bi-funnel me-2"></i>Filtrar por Status</h5>
            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
          </div>
          <div class="modal-body">
            <div class="d-flex justify-content-between align-items-center mb-2">
              <strong>Mostrar status</strong>
              <div class="btn-group btn-group-sm">
                <button type="button" class="btn btn-outline-light" id="btnAll">Todos</button>
                <button type="button" class="btn btn-outline-light" id="btnNone">Nenhum</button>
              </div>
            </div>
            <div class="row">
              <?php foreach ($status_all as $val => $label): ?>
                <div class="col-12">
                  <div class="form-check">
                    <input class="form-check-input st-check" type="checkbox"
                           value="<?= $val ?>" id="st-<?= $val ?>"
                           name="status[]"
                           <?= in_array($val, $sel_status, true) ? 'checked' : '' ?>>
                    <label class="form-check-label" for="st-<?= $val ?>"><?= htmlspecialchars($label) ?></label>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
          </div>
          <div class="modal-footer">
            <button class="btn btn-primary" type="submit"><i class="bi bi-check2-circle me-1"></i>Aplicar</button>
          </div>
        </div>
      </div>
    </div>
  </form>

  <!-- KPIs -->
  <div class="d-flex flex-wrap gap-2 mb-3">
    <?php foreach ($status_all as $st => $label): ?>
      <div class="kpi">
        <div class="label"><?= htmlspecialchars($label) ?></div>
        <div class="value"><?= (int)($kpiMap[$st] ?? 0) ?></div>
      </div>
    <?php endforeach; ?>
  </div>

  <!-- Lista -->
  <div class="card">
    <div class="table-responsive">
      <table class="table table-dark table-hover align-middle mb-0">
        <thead>
          <tr>
            <th style="width:80px">ID</th>
            <th style="min-width:260px">Cliente</th>
            <th style="width:160px">Status</th>
            <th style="min-width:280px">Última nota</th>
            <th style="width:170px">Próx. ação</th>
            <th style="width:170px">Atualizado</th>
            <th style="width:180px" class="text-end">Ações</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r):
            $canUpd = canRow('update', $r, $me_id);
            $canDel = canRow('delete', $r, $me_id);
          ?>
          <tr>
            <td><?= (int)$r['id'] ?></td>
            <td>
              <strong><?= htmlspecialchars($r['cliente_nome'] ?? '-') ?></strong>
              <?php if (!empty($r['owner_nome'])): ?>
                <div class="small-muted"><i class="bi bi-person-badge"></i> Resp.: <?= htmlspecialchars($r['owner_nome']) ?></div>
              <?php endif; ?>
            </td>
            <td>
              <span class="badge bg-<?= badgeStatus($r['status'] ?? 'novo') ?>">
                <?= htmlspecialchars($status_all[$r['status']] ?? ucfirst($r['status'] ?? 'novo')) ?>
              </span>
            </td>
            <td>
              <?php if (!empty($r['last_nota'])): ?>
                <div class="text-truncate-2" title="<?= htmlspecialchars($r['last_nota']) ?>">
                  <?= htmlspecialchars($r['last_nota']) ?>
                </div>
                <div class="small-muted"><i class="bi bi-clock"></i> <?= format_date($r['last_log_at'] ?? null, 'd/m H:i') ?></div>
              <?php else: ?>
                <span class="text-muted">—</span>
              <?php endif; ?>
            </td>
            <td><?= !empty($r['next_followup_at']) ? format_date($r['next_followup_at'], 'd/m/Y H:i') : '—' ?></td>
            <td><?= format_date($r['updated_ref'] ?? null, 'd/m/Y H:i') ?></td>
            <td class="text-end">
              <div class="btn-group">
                <a class="btn btn-sm btn-outline-light" href="<?=$BASE?>/atendimentos/visualizar?id=<?=(int)$r['id']?>">
                  <i class="bi bi-eye"></i>
                </a>
                <?php if ($canUpd): ?>
                <a class="btn btn-sm btn-outline-success" href="<?=$BASE?>/atendimentos/editar?id=<?=(int)$r['id']?>">
                  <i class="bi bi-pencil"></i>
                </a>
                <?php endif; ?>
                <?php if ($canDel): ?>
                <a class="btn btn-sm btn-outline-danger" href="<?=$BASE?>/atendimentos/remover?id=<?=(int)$r['id']?>" onclick="return confirm('Remover este atendimento?');">
                  <i class="bi bi-trash"></i>
                </a>
                <?php endif; ?>
              </div>
            </td>
          </tr>
          <?php endforeach; if (!$rows): ?>
          <tr><td colspan="7" class="text-center text-muted py-4">Nenhum atendimento encontrado</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script>
(function(){
  const checks = Array.from(document.querySelectorAll('.st-check'));
  const badge  = document.getElementById('badgeCount');
  const btnAll = document.getElementById('btnAll');
  const btnNone= document.getElementById('btnNone');

  function updateBadge(){
    if (!badge) return;
    const total  = checks.length;
    const marked = checks.filter(c => c.checked).length;
    if (marked === 0) {
      badge.textContent = '0';
      badge.className = 'badge bg-dark ms-1 badge-count';
    } else if (marked === total) {
      badge.textContent = 'Todos';
      badge.className = 'badge bg-secondary ms-1 badge-count';
    } else {
      badge.textContent = marked;
      badge.className = 'badge bg-secondary ms-1 badge-count';
    }
  }

  checks.forEach(c => c.addEventListener('change', updateBadge));
  if (btnAll)  btnAll.addEventListener('click', () => { checks.forEach(c => c.checked = true);  updateBadge(); });
  if (btnNone) btnNone.addEventListener('click', () => { checks.forEach(c => c.checked = false); updateBadge(); });

  updateBadge();
})();
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>