<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('atendimentos','view');

$BASE   = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id  = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$scopeV = rbac_scope('atendimentos','view'); // 'own'|'all'

$client_id = (int)($_GET['client_id'] ?? 0);
if ($client_id <= 0) { header("Location: {$BASE}/atendimentos"); exit; }

// Buscar cliente
$cliente = fetch_one("SELECT id, nome_completo, owner_user_id, created_by FROM clients WHERE id = ?", [$client_id], true);
if (!$cliente) { header("Location: {$BASE}/atendimentos"); exit; }

// Se escopo 'own', validar dono do cliente (owner_user_id ou created_by)
if ($scopeV === 'own') {
  $owner = (int)($cliente['owner_user_id'] ?? 0);
  $fallback = (int)($cliente['created_by'] ?? 0);
  if (!in_array($me_id, [$owner, $fallback], true)) {
    http_response_code(403); exit('Você não tem permissão para ver este cliente.');
  }
}

/* ===== CSRF ===== */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* ===== POST: criar interação ===== */
$erro = null; $ok = false;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['nova_interacao'])) {
  $valid_status = ['novo','em_atendimento','negociacao','finalizado'];
  $st  = $_POST['status'] ?? 'novo';
  $note = trim($_POST['note'] ?? '');
  $next = trim($_POST['next_action_at'] ?? '');
  $csrf = $_POST['csrf'] ?? '';

  if (!hash_equals($CSRF, (string)$csrf)) {
    $erro = 'CSRF inválido.';
  } elseif (!in_array($st, $valid_status, true)) {
    $erro = 'Status inválido.';
  } else {
    try {
      $next_dt = $next !== '' ? $next : null;
      execute("INSERT INTO interactions (client_id, owner_user_id, status, note, next_action_at, created_by) VALUES (?,?,?,?,?,?)",
        [$client_id, $me_id ?: null, $st, $note ?: null, $next_dt, $me_id ?: null]);
      $ok = true;
    } catch(Throwable $e) {
      $erro = 'Erro ao registrar: '.$e->getMessage();
    }
  }
}

/* ===== Buscar interações ===== */
$params = [$client_id];
$whereI = "WHERE i.client_id = ?";
if ($scopeV === 'own') { $whereI .= " AND i.owner_user_id = ?"; $params[] = $me_id; }

$its = fetch_all("
  SELECT i.*, u.nome AS owner_nome
  FROM interactions i
  LEFT JOIN users u ON u.id = i.owner_user_id
  $whereI
  ORDER BY i.created_at DESC, i.id DESC
", $params, true) ?: [];

$page_title = 'Atendimento: '.htmlspecialchars($cliente['nome_completo'] ?? ('#'.$client_id));
include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';

function badgeStatus($s) {
  $map = [
    'novo'            => 'secondary',
    'em_atendimento'  => 'info',
    'negociacao'      => 'warning',
    'finalizado'      => 'success'
  ];
  return $map[$s] ?? 'secondary';
}
function format_date($dt,$fmt='d/m/Y H:i'){ return $dt ? date($fmt, strtotime($dt)) : '-'; }
?>
<style>
.timeline { border-left:2px solid rgba(255,255,255,.15); margin-left: .75rem; padding-left: 1rem; }
.ti { position:relative; margin-bottom:1rem; }
.ti::before { content:''; position:absolute; left:-13px; top:4px; width:10px; height:10px; border-radius:50%; background:#6c757d; }
.ti .head { display:flex; gap:.5rem; align-items:center; }
.ti .body { padding:.35rem .5rem; background:#101114; border:1px solid rgba(255,255,255,.08); border-radius:.35rem; margin-top:.35rem; }
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div class="d-flex align-items-center gap-2">
      <i class="bi bi-clock-history" style="font-size:1.6rem; color: var(--primary);"></i>
      <h1 class="h4 m-0">Atendimento — <?= htmlspecialchars($cliente['nome_completo']) ?></h1>
    </div>
    <a href="<?=$BASE?>/atendimentos" class="btn btn-outline-secondary"><i class="bi bi-arrow-left"></i> Voltar</a>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i> Interação registrada com sucesso!
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>
  <?php if ($erro): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i> <?= htmlspecialchars($erro) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if (rbac_can('atendimentos','create')): ?>
  <div class="card mb-3">
    <div class="card-header">Nova interação</div>
    <div class="card-body">
      <form method="post" class="row g-2">
        <input type="hidden" name="nova_interacao" value="1">
        <input type="hidden" name="csrf" value="<?= htmlspecialchars($CSRF) ?>">
        <div class="col-md-3">
          <label class="form-label">Status</label>
          <select name="status" class="form-select" required>
            <option value="novo">Novo</option>
            <option value="em_atendimento">Em Atendimento</option>
            <option value="negociacao">Negociação</option>
            <option value="finalizado">Finalizado</option>
          </select>
        </div>
        <div class="col-md-4">
          <label class="form-label">Próxima ação (opcional)</label>
          <input type="datetime-local" name="next_action_at" class="form-control">
        </div>
        <div class="col-md-12">
          <label class="form-label">Anotações (opcional)</label>
          <textarea name="note" class="form-control" rows="3" placeholder="Detalhes do contato, próximos passos, objeções..."></textarea>
        </div>
        <div class="col-md-12 d-flex justify-content-end gap-2">
          <button class="btn btn-primary"><i class="bi bi-check2-circle"></i> Registrar</button>
        </div>
      </form>
    </div>
  </div>
  <?php endif; ?>

  <div class="card">
    <div class="card-header">Linha do tempo</div>
    <div class="card-body">
      <?php if (!$its): ?>
        <div class="text-muted">Sem interações registradas.</div>
      <?php else: ?>
        <div class="timeline">
          <?php foreach ($its as $it): ?>
          <div class="ti">
            <div class="head">
              <span class="badge bg-<?=badgeStatus($it['status'])?>"><?=ucfirst(str_replace('_',' ',$it['status']))?></span>
              <small class="text-muted"><i class="bi bi-calendar-event"></i> <?= format_date($it['created_at']) ?></small>
              <?php if (!empty($it['next_action_at'])): ?>
                <small class="text-warning"><i class="bi bi-flag"></i> Próx.: <?= format_date($it['next_action_at'], 'd/m H:i') ?></small>
              <?php endif; ?>
              <small class="text-muted ms-auto"><i class="bi bi-person"></i> <?= htmlspecialchars($it['owner_nome'] ?? '-') ?></small>
            </div>
            <?php if (!empty($it['note'])): ?>
            <div class="body"><?= nl2br(htmlspecialchars($it['note'])) ?></div>
            <?php endif; ?>
          </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>
