<?php
// modules/agenda/calendario.php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Agenda — Calendário';
$active     = 'agenda';

rbac_require('agenda','view');

$BASE      = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id     = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$scopeView = rbac_scope('agenda','view'); // 'own' | 'all'

// ===== Eventos (respeita escopo) =====
$params = [];
$sql = "SELECT a.id, a.titulo, a.inicio, a.fim, a.obs, a.owner_user_id,
               p.titulo AS property_titulo,
               c.nome_completo AS cliente_nome,
               u.nome AS responsavel_nome, u.email AS responsavel_email
        FROM agenda a
        LEFT JOIN properties p ON p.id = a.property_id
        LEFT JOIN clients    c ON c.id = a.client_id
        LEFT JOIN users      u ON u.id = a.owner_user_id";
if ($scopeView === 'own') { $sql .= " WHERE a.owner_user_id = ?"; $params[] = $me_id; }
$sql .= " ORDER BY a.inicio DESC LIMIT 500";

$rows = [];
try { $rows = fetch_all($sql, $params, true) ?: []; } catch (Throwable $e) { $rows = []; }

$events = [];
$can_update = rbac_can('agenda','update');
$upd_scope  = rbac_scope('agenda','update'); // 'own' | 'all'
foreach ($rows as $r) {
  $bits = [];
  if (!empty($r['titulo']))          $bits[] = $r['titulo'];
  if (!empty($r['cliente_nome']))    $bits[] = $r['cliente_nome'];
  if (!empty($r['property_titulo'])) $bits[] = $r['property_titulo'];

  // responsável (para tooltip e label)
  $resp = trim((string)($r['responsavel_nome'] ?: ''));
  if (!$resp && !empty($r['responsavel_email'])) $resp = $r['responsavel_email'];

  $title = implode(' • ', $bits) ?: ('#'.$r['id']);
  $allow_edit = false;
  if ($can_update) {
    if     ($upd_scope === 'all') $allow_edit = true;
    elseif ($upd_scope === 'own' && (int)$r['owner_user_id'] === $me_id) $allow_edit = true;
  }
  $url = $allow_edit ? ($BASE . '/agenda/editar?id=' . (int)$r['id']) : ($BASE . '/agenda');

  $events[] = [
    'id'    => (int)$r['id'],
    'title' => $title,
    'start' => $r['inicio'],
    'end'   => $r['fim'] ?: null,
    'url'   => $url,
    'extendedProps' => [
      'ownerId'   => (int)$r['owner_user_id'],
      'ownerName' => $resp,
    ],
    'className' => ((int)$r['owner_user_id'] === $me_id) ? 'evt-own' : 'evt-other',
  ];
}
$events_json = json_encode($events, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);

// ===== Assets + inicialização =====
$extra_js = <<<JS
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.11/main.min.css">
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.11/index.global.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/@fullcalendar/bootstrap5@6.1.11/index.global.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
  const el = document.getElementById('calendar');
  const titleEl = document.getElementById('calTitle');
  const btnPrev  = document.getElementById('btnPrev');
  const btnNext  = document.getElementById('btnNext');
  const btnToday = document.getElementById('btnToday');
  const btnMonth = document.getElementById('vMonth');
  const btnWeek  = document.getElementById('vWeek');
  const btnDay   = document.getElementById('vDay');
  const btnList  = document.getElementById('vList');

  const calendar = new FullCalendar.Calendar(el, {
    locale: 'pt-br',
    themeSystem: 'bootstrap5',
    initialView: 'dayGridMonth',
    height: 'auto',
    headerToolbar: false,
    buttonText: { today:'Hoje', month:'mês', week:'semana', day:'dia', list:'lista' },
    dayMaxEventRows: 3,
    nowIndicator: true,
    eventTimeFormat: { hour: '2-digit', minute: '2-digit', hour12: false },
    events: $events_json,

    datesSet: function(){ titleEl.textContent = calendar.view.title; setActive(calendar.view.type); },

    eventClassNames: (arg) => arg.event.classNames,

    eventDidMount: function(info){
      // Tooltip nativo com responsável (quando houver)
      const owner = info.event.extendedProps.ownerName;
      if (owner) {
        info.el.setAttribute('title', 'Responsável: ' + owner);
      }
    },

    eventClick: function(info) {
      if (info.event.url) { info.jsEvent.preventDefault(); window.location.href = info.event.url; }
    }
  });

  function setActive(type){
    [btnMonth,btnWeek,btnDay,btnList].forEach(b=>b.classList.remove('active'));
    if (type === 'dayGridMonth') btnMonth.classList.add('active');
    if (type === 'timeGridWeek') btnWeek.classList.add('active');
    if (type === 'timeGridDay')  btnDay.classList.add('active');
    if (type === 'listWeek')     btnList.classList.add('active');
  }

  btnPrev.addEventListener('click', ()=>calendar.prev());
  btnNext.addEventListener('click', ()=>calendar.next());
  btnToday.addEventListener('click', ()=>calendar.today());
  btnMonth.addEventListener('click', ()=>calendar.changeView('dayGridMonth'));
  btnWeek.addEventListener('click', ()=>calendar.changeView('timeGridWeek'));
  btnDay.addEventListener('click',  ()=>calendar.changeView('timeGridDay'));
  btnList.addEventListener('click', ()=>calendar.changeView('listWeek'));

  calendar.render();
});
</script>
JS;

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
  /* ========= LUXO SUAVE ========= */
  .cal-toolbar{
    display:flex; gap:.75rem; align-items:center; justify-content:space-between;
    padding:.75rem .9rem .5rem .9rem;
  }
  .cal-toolbar .btn{
    border-radius:.6rem;
    backdrop-filter: saturate(120%) blur(2px);
  }
  .cal-toolbar h2{
    letter-spacing:.2px;
    color: var(--text-secondary);
  }

  /* Segmentação com glow sutil */
  .segmented .btn{
    border-radius:.7rem!important;
    font-weight:700;
    text-transform:uppercase;
    letter-spacing:.02em;
    color:#dfe8e6 !important;
    border-color: rgba(25,135,84,.35) !important;
    background: linear-gradient(180deg, rgba(25,135,84,.14), rgba(25,135,84,.10)) !important;
  }
  .segmented .btn:hover{
    background: rgba(25,135,84,.22) !important;
    border-color: rgba(25,135,84,.55) !important;
  }
  .segmented .btn.active{
    background: linear-gradient(180deg, rgba(25,135,84,.38), rgba(25,135,84,.28)) !important;
    border-color: var(--bs-success) !important;
    color:#fff !important;
    box-shadow: 0 0 0 2px rgba(25,135,84,.25);
  }

  /* ======== FullCalendar — dark chic ======== */
  .fc-theme-bootstrap5 .fc-scrollgrid,
  .fc-theme-bootstrap5 .fc-list,
  .fc-theme-bootstrap5 .fc-timegrid-slot,
  .fc-theme-bootstrap5 .fc-daygrid-day { border-color: rgba(255,255,255,.10); }

  /* Cabeçalhos dos dias (mês/semana/dia) mais discretos */
  .fc-theme-bootstrap5 .fc-col-header-cell{
    background: var(--bg-card) !important;
    color: var(--text-secondary) !important;
    border-color: rgba(255,255,255,.06) !important;
    border-bottom: 2px solid rgba(25,135,84,.35) !important; /* acento */
  }
  .fc-theme-bootstrap5 .fc-col-header-cell a.fc-col-header-cell-cushion{
    color: var(--text-secondary) !important;
    text-decoration:none;
    font-weight:700;
  }

  /* Hoje: aro suave */
  .fc .fc-daygrid-day.fc-day-today .fc-daygrid-day-frame,
  .fc .fc-timegrid-col.fc-day-today .fc-timegrid-col-frame{
    box-shadow: inset 0 0 0 2px rgba(25,135,84,.35);
    background: rgba(25,135,84,.06) !important;
  }

  /* Fim de semana com leve faixa */
  .fc-theme-bootstrap5 .fc-day-sat .fc-daygrid-day-frame,
  .fc-theme-bootstrap5 .fc-day-sun .fc-daygrid-day-frame{ background: rgba(255,255,255,.02); }

  .fc-theme-bootstrap5 .fc-daygrid-day-number{ color: rgba(255,255,255,.75); font-weight:600; }

  /* ===== EVENTOS (mês/semana/dia) — pill com borda lateral ===== */
  .fc-theme-bootstrap5 .fc-event,
  .fc-theme-bootstrap5 .fc-daygrid-event{
    border-radius:.55rem; padding:2px 8px; border:1px solid rgba(255,255,255,.12);
    background: var(--bg-card-hover); color:#e6edf3; 
    box-shadow: 0 2px 10px rgba(0,0,0,.15);
  }
  .fc .evt-own .fc-event-main, .fc .evt-own { border-left:3px solid var(--bs-success) !important; }
  .fc .evt-other .fc-event-main, .fc .evt-other { border-left:3px solid var(--bs-info) !important; }
  .fc-theme-bootstrap5 .fc-event:hover{ background: #21262d; border-color: rgba(255,255,255,.18); }

  /* NOW indicator mais elegante */
  .fc .fc-now-indicator-line{ border-color: rgba(255, 92, 92, .9); }
  .fc .fc-now-indicator-arrow{ border-color: transparent rgba(255, 92, 92, .9) transparent transparent; }

  /* ===================== LIST VIEW ===================== */

  /* Cabeçalho do dia na lista — escuro com acento, sem “tudo verde” */
  .fc .fc-list .fc-list-day-cushion,
  .fc .fc-list-sticky .fc-list-day-cushion{
    background: var(--bg-card-hover) !important;
    color: var(--text-secondary) !important;
    border-top: 1px solid rgba(255,255,255,.08) !important;
    border-bottom: 2px solid rgba(25,135,84,.35) !important;
  }
  .fc .fc-list-day-text,
  .fc .fc-list-day-side-text{ color: var(--text-secondary) !important; font-weight:700; }

  .fc-theme-bootstrap5 .fc-list{
    background: transparent !important;
    color: var(--text, #e6edf3) !important;
    border-color: rgba(255,255,255,.10) !important;
  }
  .fc-theme-bootstrap5 .fc-list-table td,
  .fc-theme-bootstrap5 .fc-list-table th{
    border-color: rgba(255,255,255,.08) !important;
  }
  /* linhas de evento escuras, com hover elegante */
  .fc-theme-bootstrap5 .fc-list-event td{ background: var(--bg-card) !important; }
  .fc-theme-bootstrap5 .fc-list-event:hover td{ background: var(--bg-card-hover) !important; }

  /* pontinho e textos */
  .fc .fc-list-event-dot{ border-color: var(--bs-success) !important; background: var(--bs-success) !important; }
  .fc-theme-bootstrap5 .fc-list-event-time{ color: #c9d1d9 !important; font-weight:600; }
  .fc-theme-bootstrap5 .fc-list-event-title a{ color: #e6edf3 !important; text-decoration:none; }

  /* Badge de responsável (aparece via title nativo; opcional colocar depois no HTML) */
</style>

<div class="container my-4">
  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center gap-2 mb-3">
    <div>
      <div class="d-flex align-items-center gap-2">
        <i class="bi bi-calendar-week" style="font-size:1.6rem; color: var(--bs-success);"></i>
        <h1 class="h4 m-0">Agenda — Calendário</h1>
      </div>
      <div class="text-muted small mt-1">Visualize e navegue pelos compromissos com um toque mais elegante</div>
    </div>
    <div class="d-flex gap-2">
      <a href="<?=$BASE?>/agenda" class="btn btn-outline-secondary">
        <i class="bi bi-list-ul"></i> Ver lista
      </a>
      <?php if (rbac_can('agenda','create')): ?>
        <a href="<?=$BASE?>/agenda/novo" class="btn btn-success">
          <i class="bi bi-plus-lg"></i> Novo compromisso
        </a>
      <?php endif; ?>
    </div>
  </div>

  <div class="card">
    <div class="cal-toolbar">
      <div class="btn-group" role="group" aria-label="Navegação">
        <button id="btnPrev" class="btn btn-outline-secondary"><i class="bi bi-chevron-left"></i></button>
        <button id="btnToday" class="btn btn-outline-secondary">Hoje</button>
        <button id="btnNext" class="btn btn-outline-secondary"><i class="bi bi-chevron-right"></i></button>
      </div>

      <h2 id="calTitle" class="h5 m-0 text-center flex-grow-1"></h2>

      <div class="btn-group segmented" role="group" aria-label="Visões">
        <button id="vMonth" class="btn active">mês</button>
        <button id="vWeek"  class="btn">semana</button>
        <button id="vDay"   class="btn">dia</button>
        <button id="vList"  class="btn">lista</button>
      </div>
    </div>
    <div class="card-body pt-2">
      <div id="calendar"></div>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>